<?php
declare(strict_types = 1);

require_once(__DIR__ . '/Admin.class.php');

/**
 * Ajout de fichiers par FTP.
 *
 * @license https://www.gnu.org/licenses/gpl-3.0.html
 * @link https://www.igalerie.org/
 */
class AdminFTP extends Admin
{
	/**
	 * Ajoute à la galerie les fichiers envoyés par FTP.
	 *
	 * @return void
	 */
	public static function addFiles(): void
	{
		if (!isset($_POST['action']) || $_POST['action'] != 'scan')
		{
			return;
		}

		// Données de template.
		$set_tpl = function(&$tpl)
		{
			foreach ($tpl as $key => &$data)
			{
				Template::set($key, $data);
			}
		};
		$tpl = ['scan' => 1];

		// Initialisation du scan.
		$scan = new Scan();
		if (!$scan->getInit)
		{
			$tpl['error'] = __('Initialisation du scan échouée.');
			$set_tpl($tpl);
		}

		// Options de scan.
		$scan->setDeleteRejectedFiles = isset($_POST['delete_rejected_files']);
		$scan->setStatus = (int) isset($_POST['files_status']);
		$scan->setUpdateFiles = FALSE;
		$scan->setUpdateThumbId = FALSE;
		$scan->setReportAllFiles = TRUE;
		$scan->setUserId = Auth::$id;
		$scan->setUserName = Auth::$nickname;

		// Groupes dont les utilisateurs ont déjà été notifié par
		// courriel lors du précédent scan qui n'a pas été complet
		// à cause de la limite de temps d'exécution.
		if (Utility::checkPost('notify_groups_exclude', '`^\d+(\,\d+)*$`'))
		{
			$scan->setNotifyGroupsExclude = explode(',', $_POST['notify_groups_exclude']);
		}

		// Lancement du scan.
		if ($scan->start() === FALSE)
		{
			$tpl['error'] = __('Une erreur s\'est produite.');
			$set_tpl($tpl);
		}

		// Contrôle du temps d'exécution.
		if ($scan->getTimeExceeded)
		{
			$tpl['notify_groups_exclude'] = implode(',', array_merge(
				$scan->setNotifyGroupsExclude,
				$scan->getNotifyGroups
			));
			$tpl['time_exceeded'] = __('Durée limite du scan dépassée,'
				. ' cliquez à nouveau sur le bouton pour scanner les albums restants.');
		}

		// Rapport de scan.
		$report =& $scan->getReport;

		// Erreurs.
		if ($report['files_errors'])
		{
			$message = count($report['files_errors']) > 1
				? __('%s erreurs se sont produites.')
				: __('%s erreur s\'est produite.');
			$tpl['error_files']
				= sprintf($message, count($report['files_errors']));
		}

		// Albums, images et/ou vidéos ajoutés.
		if ((count($report['albums_added'])
		+ $report['images_added'] + $report['videos_added']) > 0)
		{
			$images = $report['images_added'] > 1
				? __('%s images')
				: __('%s image');
			$images = sprintf($images, $report['images_added']);
			$videos = $report['videos_added'] > 1
				? __('%s vidéos')
				: __('%s vidéo');
			$videos = sprintf($videos, $report['videos_added']);

			// Albums + images et/ou vidéos.
			if (count($report['albums_added']) > 0)
			{
				$albums = count($report['albums_added']) > 1
					? __('%s albums')
					: __('%s album');
				$albums = sprintf($albums, count($report['albums_added']));
				$files = ($report['images_added'] > 0 && $report['videos_added'] > 0)
					? $albums . ', ' . sprintf('%s et %s', $images, $videos)
					: sprintf(
						__('%s et %s'),
						$albums,
						(($report['images_added'] > 0) ? $images : $videos)
					  );
				$message = sprintf(__('%s ont été ajoutés à la base de données.'), $files);
			}

			// Images + vidéos.
			else if ($report['images_added'] > 0 && $report['videos_added'] > 0)
			{
				$message = sprintf(
					__('%s ont été ajoutées à la base de données.'),
					sprintf(__('%s et %s'), $images, $videos)
				);
			}

			// Images.
			else if ($report['images_added'] > 0)
			{
				$message = $report['images_added'] > 1
					? __('%s ont été ajoutées à la base de données.')
					: __('%s a été ajoutée à la base de données.');
				$message = sprintf($message, $images);
			}

			// Vidéos.
			else
			{
				$message = $report['videos_added'] > 1
					? __('%s ont été ajoutées à la base de données.')
					: __('%s a été ajoutée à la base de données.');
				$message = sprintf($message, $videos);
			}

			$tpl['added_files'] = $message;
			$tpl['datetime_link'] = App::getURL(
				'category/1/datetime/' . strtotime($scan->getNow)
			);
		}
		else
		{
			$tpl['added_none'] =
				__('Aucun nouvel album et aucun nouveau fichier n\'a été détecté.');
		}

		// Albums, images et/ou vidéos mis à jour.
		if ((count($report['albums_updated'])
		+ $report['images_updated'] + $report['videos_updated']) > 0)
		{
			$albums = count($report['albums_updated']) > 1
				? __('%s albums')
				: __('%s album');
			$albums = sprintf($albums, count($report['albums_updated']));

			// Albums + images et/ou vidéos.
			if ($report['images_updated'] > 0 || $report['videos_updated'] > 0)
			{
				$images = $report['images_updated'] > 1
					? __('%s images')
					: __('%s image');
				$images = sprintf($images, $report['images_updated']);
				$videos = $report['videos_updated'] > 1
					? __('%s vidéos')
					: __('%s vidéo');
				$videos = sprintf($videos, $report['videos_updated']);
				$files = ($report['images_updated'] > 0 && $report['videos_updated'] > 0)
					? $albums . ', ' . sprintf('%s et %s', $images, $videos)
					: sprintf(
						__('%s et %s'),
						$albums,
						(($report['images_updated'] > 0) ? $images : $videos)
					  );
				$message = sprintf(__('%s ont été mis à jour.'), $files);
			}

			// Albums.
			else
			{
				$message = count($report['albums_updated']) > 1
					? __('%s ont été mis à jour.')
					: __('%s a été mis à jour.');
				$message = sprintf($message, $albums);
			}

			$tpl['updated_files'] = $message;
		}
		else
		{
			$tpl['updated_none'] = __('Aucun album n\'a été mis à jour.');
		}

		// Fichiers et répertoires rejetés.
		if ((count($report['dirs_rejected']) + count($report['files_rejected'])) > 0)
		{
			if (count($report['dirs_rejected']) > 0 && count($report['files_rejected']) > 0)
			{
				$categories = count($report['dirs_rejected']) > 1
					? __('%s catégories')
					: __('%s catégorie');
				$categories = sprintf($categories, count($report['dirs_rejected']));
				$items = count($report['files_rejected']) > 1
					? __('%s fichiers')
					: __('%s fichier');
				$items = sprintf($items, count($report['files_rejected']));
				$message = $scan->setDeleteRejectedFiles
					? __('%s et %s ont été rejetés et supprimés.')
					: __('%s et %s ont été rejetés.');
				$message = sprintf($message, $categories, $items);
			}
			else if (count($report['dirs_rejected']) > 0)
			{
				$message = $scan->setDeleteRejectedFiles
					? (count($report['dirs_rejected']) > 1
						? __('%s catégories ont été rejetées et supprimées.')
						: __('%s catégorie a été rejetée et supprimée.'))
					: (count($report['dirs_rejected']) > 1
						? __('%s catégories ont été rejetées.')
						: __('%s catégorie a été rejetée.'));
				$message = sprintf($message, count($report['dirs_rejected']));
			}
			else
			{
				$message = $scan->setDeleteRejectedFiles
					? (count($report['files_rejected']) > 1
						? __('%s fichiers ont été rejetés et supprimés.')
						: __('%s fichier a été rejeté et supprimé.'))
					: (count($report['files_rejected']) > 1
						? __('%s fichiers ont été rejetés.')
						: __('%s fichier a été rejeté.'));
				$message = sprintf($message, count($report['files_rejected']));
			}
			$tpl['rejected'] = $message;
		}

		// Rapport détaillé : erreurs et fichiers rejetés.
		foreach (['dirs_rejected', 'files_errors', 'files_rejected'] as $i)
		{
			if ($report[$i])
			{
				$tpl['details'][$i] = $report[$i];
			}
		}

		// Rapport détaillé : albums ajoutés et mis à jour.
		foreach (['added', 'updated'] as $i)
		{
			if ($report['albums_' . $i])
			{
				$tpl['details']['albums_' . $i] = [];
				foreach ($report['albums_' . $i] as &$v)
				{
					$tpl['details']['albums_' . $i][] =
					[
						'album' => $v['album'],
						'nb_images' => $v['images'],
						'nb_videos' => $v['videos'],
						'filesize' => L10N::formatFilesize($v['size'])
					];
				}
			}
		}

		$set_tpl($tpl);
	}
}
?>