<?php
/**
 * @copyright   &copy; 2005-2024 PHPBoost
 * @license     https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL-3.0
 * @author      Julien BRISWALTER <j1.seth@phpboost.com>
 * @version     PHPBoost 6.0 - last update: 2022 01 15
 * @since       PHPBoost 4.1 - 2015 05 20
 * @contributor Arnaud GENET <elenwii@phpboost.com>
 * @contributor mipel <mipel@phpboost.com>
 * @contributor Sebastien LARTIGUE <babsolune@phpboost.com>
 * @contributor janus57 <janus57@janus57.fr>
*/

class AdminServerSystemReportController extends DefaultAdminController
{
	public function execute(HTTPRequestCustom $request)
	{
		$this->build_form();

		$this->view->put('CONTENT', $this->form->display());

		return new AdminServerDisplayResponse($this->view, $this->lang['admin.system.report']);
	}

	private function build_form()
	{
		$picture_yes = '<i class="fa fa-check fa-2x success" aria-hidden="true"></i><span class="sr-only">' . $this->lang['common.yes'] . '</span>';
		$picture_no = '<i class="fa fa-times fa-2x error" aria-hidden="true"></i><span class="sr-only">' . $this->lang['common.no'] . '</span>';
		$picture_unknown = '<i class="fa fa-question fa-2x question" aria-hidden="true"></i><span class="sr-only">' . $this->lang['common.unknown'] . '</span>';

		$default_lang_config = LangsManager::get_lang(LangsManager::get_default_lang())->get_configuration();
		$default_theme_config = ThemesManager::get_theme(ThemesManager::get_default_theme())->get_configuration();
		$editors = AppContext::get_content_formatting_service()->get_available_editors();
		$default_editor = $editors[ContentFormattingConfig::load()->get_default_editor()];
		$server_configuration = new ServerConfiguration();
		$general_config = GeneralConfig::load();
		$server_environment_config = ServerEnvironmentConfig::load();
		$sessions_config = SessionsConfig::load();

		$url_rewriting_available = false;
		$url_rewriting_known = true;

		try
		{
			$url_rewriting_available = $server_configuration->has_url_rewriting();
		}
        catch(UnsupportedOperationException $ex)
		{
			$url_rewriting_known = false;
		}

		$summerization =
"---------------------------------System report---------------------------------
-----------------------------generated by PHPBoost-----------------------------
SERVER CONFIGURATION-----------------------------------------------------------
php version			: " . ServerConfiguration::get_phpversion() . "
dbms version			: " . PersistenceContext::get_dbms_utils()->get_dbms_version() . "
gd library			: " . (int)$server_configuration->has_gd_library() . "
curl extension			: " . (int)$server_configuration->has_curl_library() . "
mbstring extension		: " . (int)$server_configuration->has_mbstring_library() . "
url rewriting			: " . ($url_rewriting_known ? (int)$url_rewriting_available : 'N/A') . "
apcu cache			: " . (int)DataStoreFactory::is_apc_available() . "
PHPBOOST CONFIGURATION---------------------------------------------------------
phpboost version		: " . Environment::get_phpboost_version() . "
server url			: " . $general_config->get_site_url() . "
site path			: " . $general_config->get_site_path() . "
default theme			: " . $default_theme_config->get_name() . " (" . $this->lang['common.version'] . " " . $default_theme_config->get_version() . ")
default language		: " . $default_lang_config->get_name() . "
default editor			: " . $default_editor . "
home page			: " . Environment::get_home_page() . "
url rewriting			: " . (int)$server_environment_config->is_url_rewriting_enabled() . "
apcu cache			: " . (int)DataStoreFactory::is_apc_enabled() . "
output gzip			: " . (int)$server_environment_config->is_output_gziping_enabled() . "
session cookie name		: " . $sessions_config->get_cookie_name() . "
session duration		: " . $sessions_config->get_session_duration() . "
active session duration		: " . $sessions_config->get_active_session_duration() . "
DIRECTORIES AUTHORIZATIONS-----------------------------------------------------
";

		$form = new HTMLForm('system-report', '', false);

		$this->get_advice($form);

		$fieldset = new FormFieldsetHTML('server-report', $this->lang['admin.server']);
		$form->add_fieldset($fieldset);

		$fieldset->add_field(new FormFieldFree('php_version', $this->lang['admin.php.version'], ServerConfiguration::get_phpversion()));
		$fieldset->add_field(new FormFieldFree('dbms_version', $this->lang['admin.dbms.version'], PersistenceContext::get_dbms_utils()->get_dbms_version()));
		$fieldset->add_field(new FormFieldFree('gd_library', $this->lang['admin.gd.library'], $server_configuration->has_gd_library() ? $picture_yes : $picture_no));
		$fieldset->add_field(new FormFieldFree('curl_library', $this->lang['admin.curl.library'], $server_configuration->has_curl_library() ? $picture_yes : $picture_no));
		$fieldset->add_field(new FormFieldFree('mbstring_library', $this->lang['admin.mbstring.library'], $server_configuration->has_mbstring_library() ? $picture_yes : $picture_no));
		$fieldset->add_field(new FormFieldFree('url_rewriting', $this->lang['admin.url.rewriting'], $url_rewriting_known ? ($url_rewriting_available ? $picture_yes : $picture_no) : $picture_unknown));
		$fieldset->add_field(new FormFieldFree('apcu_cache', $this->lang['admin.apcu.cache'], DataStoreFactory::is_apc_available() ? $picture_yes : $picture_no));

		$fieldset = new FormFieldsetHTML('phpboost-config-report', $this->lang['admin.phpboost.config']);
		$form->add_fieldset($fieldset);

		$fieldset->add_field(new FormFieldFree('kernel_version', $this->lang['admin.kernel.version'], Environment::get_phpboost_version()));
		$fieldset->add_field(new FormFieldFree('site_url', $this->lang['configuration.site.url'], $general_config->get_site_url()));
		$fieldset->add_field(new FormFieldFree('site_path', $this->lang['configuration.site.path'], $general_config->get_site_path()));
		$fieldset->add_field(new FormFieldFree('default_theme', $this->lang['configuration.default.theme'], $default_theme_config->get_name() . " (" . $this->lang['common.version'] . " " . $default_theme_config->get_version() . ")"));
		$fieldset->add_field(new FormFieldFree('default_language', $this->lang['configuration.default.language'], $default_lang_config->get_name()));
		$fieldset->add_field(new FormFieldFree('default_editor', $this->lang['admin.default.formatting.language'], $default_editor));
		$fieldset->add_field(new FormFieldFree('start_page', $this->lang['configuration.start.page'], Environment::get_home_page()));
		$fieldset->add_field(new FormFieldFree('phpboost_url_rewriting', $this->lang['admin.url.rewriting'], $server_environment_config->is_url_rewriting_enabled() ? $picture_yes : $picture_no));
		$fieldset->add_field(new FormFieldFree('phpboost_apcu_cache', $this->lang['admin.apcu.cache'], DataStoreFactory::is_apc_enabled() ? $picture_yes : $picture_no));
		$fieldset->add_field(new FormFieldFree('output_gz', $this->lang['admin.output.gz'], $server_environment_config->is_output_gziping_enabled() ? $picture_yes : $picture_no));
		$fieldset->add_field(new FormFieldFree('cookie_name', $this->lang['configuration.cookie.name'], $sessions_config->get_cookie_name()));
		$fieldset->add_field(new FormFieldFree('session_length', $this->lang['configuration.cookie.duration'], $sessions_config->get_session_duration()));
		$fieldset->add_field(new FormFieldFree('session_guest_length', $this->lang['configuration.active.session.duration'], $sessions_config->get_active_session_duration()));

		$fieldset = new FormFieldsetHTML('directories_auth', $this->lang['admin.directories.auth']);
		$form->add_fieldset($fieldset);

		$directories_summerization = '';
		foreach(PHPBoostFoldersPermissions::get_permissions() as $key => $folder)
		{
			$fieldset->add_field(new FormFieldFree(str_replace('/', '_', $key), $key, $folder->is_writable() ? $picture_yes : $picture_no));
			$directories_summerization .= $key . str_repeat('	', 5 - (int)(TextHelper::strlen($key) / 8)) . ": " . (int)$folder->is_writable() . "
";
		}

		$fieldset = new FormFieldsetHTML('summerization', $this->lang['admin.system.report.summary']);
		$form->add_fieldset($fieldset);

		$fieldset->add_field(new FormFieldLabel($this->lang['admin.system.report.summary.clue'],
			array('class' => 'full-field')
		));

		$fieldset->add_element(new FormButtonButton('<i class="far fa-fw fa-copy"></i>' . $this->lang['admin.copy.report'], 'copy_code_clipboard(\'system-report_report-content\')', 'copy_report', 'copy-to-clipboard', '', 'system-report'));

		$fieldset->add_field(new FormFieldMultiLineTextEditor('report-content', '', $summerization . $directories_summerization,
			array('rows' => 20, 'cols' => 15, 'class' => 'system-report')
		));

		$this->form = $form;
	}

	public static function get_advice(HTMLForm $html_form)
	{
		$lang = LangLoader::get_all_langs();

		$server_configuration = new ServerConfiguration();
		$maintenance_config = MaintenanceConfig::load();
		$general_config = GeneralConfig::load();
		$server_environment_config = ServerEnvironmentConfig::load();
		$security_config = SecurityConfig::load();

		$url_rewriting_available = false;
		try
		{
			$url_rewriting_available = $server_configuration->has_url_rewriting();
		} catch (UnsupportedOperationException $ex) {}

		$fieldset = new FormFieldsetHTML('advice', $lang['admin.advice']);

		if (ModulesManager::is_module_installed('QuestionCaptcha') && ModulesManager::is_module_activated('QuestionCaptcha') && ContentManagementConfig::load()->get_used_captcha_module() == 'QuestionCaptcha' && QuestionCaptchaConfig::load()->count_items() < 3)
			$fieldset->add_field(new FormFieldFree('QuestionCaptcha_questions_number', '', MessageHelper::display(LangLoader::get_message('advices.questioncaptcha.items.number', 'common', 'QuestionCaptcha'), MessageHelper::WARNING)->render()));

		$fieldset->add_field(new FormFieldFree('modules_management', '', MessageHelper::display($lang['admin.advice.modules.management'], MessageHelper::SUCCESS)->render()));

		if ($maintenance_config->is_under_maintenance())
			$fieldset->add_field(new FormFieldFree('check_modules_authorizations', '', MessageHelper::display($lang['admin.advice.check.modules.authorizations'], MessageHelper::SUCCESS)->render()));

		if (!TextHelper::strstr($general_config->get_site_url(), 'localhost') && !TextHelper::strstr($general_config->get_site_url(), '127.0.0.1') && !$maintenance_config->is_under_maintenance() && Debug::is_debug_mode_enabled())
			$fieldset->add_field(new FormFieldFree('disable_debug_mode', '', MessageHelper::display($lang['admin.advice.disable.debug.mode'], MessageHelper::WARNING)->render()));

		if ($maintenance_config->is_under_maintenance())
			$fieldset->add_field(new FormFieldFree('disable_maintenance', '', MessageHelper::display($lang['admin.advice.disable.maintenance'], MessageHelper::NOTICE)->render()));

		if ($url_rewriting_available && !$server_environment_config->is_url_rewriting_enabled())
			$fieldset->add_field(new FormFieldFree('enable_url_rewriting', '', MessageHelper::display($lang['admin.advice.enable.url.rewriting'], MessageHelper::NOTICE)->render()));

		if (function_exists('ob_gzhandler') && @extension_loaded('zlib') && !$server_environment_config->is_output_gziping_enabled())
			$fieldset->add_field(new FormFieldFree('enable_output_gz', '', MessageHelper::display($lang['admin.advice.enable.output.gz'], MessageHelper::NOTICE)->render()));

		if (DataStoreFactory::is_apc_available() && !DataStoreFactory::is_apc_enabled())
			$fieldset->add_field(new FormFieldFree('enable_apcu_cache', '', MessageHelper::display($lang['admin.advice.enable.apcu.cache'], MessageHelper::NOTICE)->render()));

		$fieldset->add_field(new FormFieldFree('save_database', '', MessageHelper::display($lang['admin.advice.save.database'], MessageHelper::SUCCESS)->render()));

		if (ModulesManager::is_module_installed('database') && ModulesManager::is_module_activated('database') && !DatabaseConfig::load()->is_database_tables_optimization_enabled())
			$fieldset->add_field(new FormFieldFree('optimize_database_tables', '', MessageHelper::display($lang['admin.advice.optimize.database.tables'], MessageHelper::SUCCESS)->render()));

		if ($security_config->get_internal_password_min_length() == 6 && $security_config->get_internal_password_strength() == SecurityConfig::PASSWORD_STRENGTH_WEAK && !$security_config->are_login_and_email_forbidden_in_password())
			$fieldset->add_field(new FormFieldFree('password_security', '', MessageHelper::display($lang['admin.advice.password.security'], MessageHelper::NOTICE)->render()));

		if (version_compare(ServerConfiguration::get_phpversion(), ServerConfiguration::RECOMMENDED_PHP_VERSION, '<'))
			$fieldset->add_field(new FormFieldFree('upgrade_php_version', '', MessageHelper::display($lang['admin.advice.upgrade.php.version'], MessageHelper::NOTICE)->render()));

		if (count($fieldset->get_fields()))
			$html_form->add_fieldset($fieldset);
	}
}
?>
