<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

use ILIAS\Wiki\Export;
use ILIAS\GlobalScreen\ScreenContext\ContextServices;
use ILIAS\Wiki\WikiGUIRequest;
use ILIAS\UI\Component\Input\Container\Form\Standard as StandardForm;
use ILIAS\Wiki\Settings\SettingsGUI;

/**
 * @author Alexander Killing <killing@leifos.de>
 *
 * @ilCtrl_Calls ilObjWikiGUI: ilPermissionGUI, ilInfoScreenGUI, ilWikiPageGUI
 * @ilCtrl_IsCalledBy ilObjWikiGUI: ilRepositoryGUI, ilAdministrationGUI
 * @ilCtrl_Calls ilObjWikiGUI: ilPublicUserProfileGUI, ilObjectContentStyleSettingsGUI
 * @ilCtrl_Calls ilObjWikiGUI: ilExportGUI, ilCommonActionDispatcherGUI
 * @ilCtrl_Calls ilObjWikiGUI: ilRatingGUI, ilWikiPageTemplateGUI, ilWikiStatGUI
 * @ilCtrl_Calls ilObjWikiGUI: ilObjectMetaDataGUI
 * @ilCtrl_Calls ilObjWikiGUI: ilSettingsPermissionGUI
 * @ilCtrl_Calls ilObjWikiGUI: ilRepositoryObjectSearchGUI, ilObjectCopyGUI, ilObjNotificationSettingsGUI
 * @ilCtrl_Calls ilObjWikiGUI: ilLTIProviderObjectSettingGUI, ilObjectTranslationGUI
 * @ilCtrl_Calls ilObjWikiGUI: ILIAS\Wiki\Settings\SettingsGUI
 */
class ilObjWikiGUI extends ilObjectGUI
{
    protected \ILIAS\Wiki\InternalDomainService $domain;
    protected \ILIAS\Wiki\InternalGUIService $gui;
    protected \ILIAS\Wiki\Content\GUIService $content_gui;
    protected \ILIAS\Wiki\Navigation\ImportantPageManager $imp_pages;
    protected \ILIAS\Wiki\Page\PageManager $pm;
    protected ilObjectTranslation $ot;
    protected \ILIAS\HTTP\Services $http;
    protected string $requested_page;
    protected ilPropertyFormGUI $form_gui;
    protected ilTabsGUI $tabs;
    protected ilHelpGUI $help;
    protected ilLogger $log;
    protected ContextServices $tool_context;
    protected \ILIAS\DI\UIServices $ui;
    protected bool $req_with_comments = false;
    protected WikiGUIRequest $edit_request;
    protected \ILIAS\Style\Content\GUIService $content_style_gui;
    protected \ILIAS\Style\Content\Object\ObjectFacade $content_style_domain;

    public function __construct(
        $a_data,
        int $a_id,
        bool $a_call_by_reference,
        bool $a_prepare_output = true
    ) {
        global $DIC;

        $gui = $DIC->wiki()->internal()->gui();
        $domain = $DIC->wiki()->internal()->domain();
        $this->gui = $gui;
        $this->domain = $domain;

        $this->ctrl = $gui->ctrl();
        $this->lng = $domain->lng();
        $this->tabs = $gui->tabs();
        $this->help = $gui->help();
        $this->locator = $gui->locator();
        $this->http = $gui->http();
        $this->ot = $gui->wiki()->translation();

        $this->type = "wiki";

        $this->log = ilLoggerFactory::getLogger('wiki');

        $this->tool_context = $gui->globalScreen()->tool()->context();
        $this->ui = $gui->ui();

        $this->edit_request = $gui->request();
        $this->content_gui = $gui->content();

        parent::__construct($a_data, $a_id, $a_call_by_reference, $a_prepare_output);
        $this->lng->loadLanguageModule("obj");
        $this->lng->loadLanguageModule("wiki");

        $this->requested_page = $this->edit_request->getPage();
        if ($this->requested_page !== "") {
            $this->ctrl->setParameter($this, "page", ilWikiUtil::makeUrlTitle($this->requested_page));
        }
        $this->ctrl->saveParameterByClass(self::class, "transl");
        $this->ctrl->saveParameterByClass(self::class, "wpg_id");
        $this->req_with_comments = $this->edit_request->getWithComments();
        $cs = $DIC->contentStyle();
        $this->content_style_gui = $cs->gui();
        if (is_object($this->object)) {
            $this->content_style_domain = $cs->domain()->styleForRefId($this->object->getRefId());
            $this->pm = $this->domain->page()->page($this->object->getRefId());
            $this->imp_pages = $this->domain->importantPage($this->object->getRefId());
        }
    }

    public function executeCommand(): void
    {
        $ilCtrl = $this->ctrl;
        $tpl = $this->tpl;
        $ilTabs = $this->tabs;
        $ilAccess = $this->access;

        $next_class = $this->ctrl->getNextClass($this);
        $cmd = $this->ctrl->getCmd();

        $this->triggerAssignmentTool();

        $this->prepareOutput();
        $this->gui->initFetch();

        // see ilWikiPageGUI::printViewOrderList()
        // printView() cannot be in ilWikiPageGUI because of stylesheet confusion
        if ($cmd === "printView") {
            $next_class = null;
        }

        switch ($next_class) {
            case "ilinfoscreengui":
                $this->checkPermission("visible");
                $this->addHeaderAction();
                $this->infoScreen();	// forwards command
                break;

            case 'ilpermissiongui':
                $this->addHeaderAction();
                $ilTabs->activateTab("perm_settings");
                $perm_gui = new ilPermissionGUI($this);
                $this->ctrl->forwardCommand($perm_gui);
                break;

            case 'ilsettingspermissiongui':
                $this->checkPermission("write");
                $this->addHeaderAction();
                $ilTabs->activateTab("settings");
                $this->setSettingsSubTabs("permission_settings");
                $perm_gui = new ilSettingsPermissionGUI($this);
                $perm_gui->setPermissions(array("edit_wiki_navigation", "add_pages", "delete_wiki_pages", "activate_wiki_protection",
                    "wiki_html_export"));
                $perm_gui->setRoleRequiredPermissions(array("edit_content"));
                $perm_gui->setRoleProhibitedPermissions(array("write"));
                $this->ctrl->forwardCommand($perm_gui);
                break;

            case 'ilobjecttranslationgui':
                $this->checkPermission("read");
                $this->addHeaderAction();
                $ilTabs->activateTab("settings");
                $this->setSettingsSubTabs("obj_multilinguality");
                $transgui = new ilObjectTranslationGUI($this);
                $transgui->setTitleDescrOnlyMode(false);
                $this->ctrl->forwardCommand($transgui);
                break;

            case 'ilwikipagegui':
                $this->checkPermission("read");
                $requested_page = $this->requested_page;
                if ($this->edit_request->getWikiPageId() > 0) {
                    $requested_page = $this->pm->getTitle(
                        $this->edit_request->getWikiPageId(),
                        $this->edit_request->getTranslation()
                    );
                }
                $wpage_gui = $this->content_gui->getCurrentPageGUI();
                $wpage_gui->setStyleId($this->content_style_domain->getEffectiveStyleId());
                $this->setContentStyleSheet();
                if (!$ilAccess->checkAccess("write", "", $this->object->getRefId()) &&
                    (
                        !$ilAccess->checkAccess("edit_content", "", $this->object->getRefId()) ||
                        $wpage_gui->getPageObject()->getBlocked()
                    )) {
                    $wpage_gui->setEnableEditing(false);
                }

                // alter title and description
                if ($ilAccess->checkAccess("write", "", $this->object->getRefId())) {
                    $wpage_gui->activateMetaDataEditor($this->object, "wpg", $wpage_gui->getId());
                }

                $ret = $this->ctrl->forwardCommand($wpage_gui);
                if ($ret != "") {
                    $tpl->setContent($ret);
                }
                break;

            case 'ilobjectcopygui':
                $cp = new ilObjectCopyGUI($this);
                $cp->setType('wiki');
                $this->ctrl->forwardCommand($cp);
                break;

            case 'ilpublicuserprofilegui':
                $profile_gui = new ilPublicUserProfileGUI(
                    $this->edit_request->getUserId()
                );
                $ret = $this->ctrl->forwardCommand($profile_gui);
                $tpl->setContent($ret);
                break;

            case "ilobjectcontentstylesettingsgui":
                $this->checkPermission("write");
                $this->addHeaderAction();
                $ilTabs->activateTab("settings");
                $this->setSettingsSubTabs("style");

                $settings_gui = $this->content_style_gui
                    ->objectSettingsGUIForRefId(
                        null,
                        $this->object->getRefId()
                    );
                $this->ctrl->forwardCommand($settings_gui);
                break;

            case "ilexportgui":
                $this->addHeaderAction();
                $ilTabs->activateTab("export");
                $exp_gui = new ilExportGUI($this);
                $this->ctrl->forwardCommand($exp_gui);
                break;

            case "ilcommonactiondispatchergui":
                $gui = ilCommonActionDispatcherGUI::getInstanceFromAjaxCall();
                $this->ctrl->forwardCommand($gui);
                break;

            case "ilratinggui":
                // for rating category editing
                $this->checkPermission("write");
                $this->addHeaderAction();
                $ilTabs->activateTab("settings");
                $this->setSettingsSubTabs("rating_categories");
                $gui = new ilRatingGUI();
                $gui->setObject($this->object->getId(), $this->object->getType());
                $gui->setExportCallback(array($this, "getSubObjectTitle"), $this->lng->txt("page"));
                $this->ctrl->forwardCommand($gui);
                break;

            case "ilwikistatgui":
                $this->checkPermission("statistics_read");

                $this->addHeaderAction();
                $ilTabs->activateTab("statistics");
                $gui = new ilWikiStatGUI($this->object->getId());
                $this->ctrl->forwardCommand($gui);
                break;

            case "ilwikipagetemplategui":
                $this->checkPermission("write");
                $this->addHeaderAction();
                $ilTabs->activateTab("settings");
                $this->setSettingsSubTabs("page_templates");
                $wptgui = new ilWikiPageTemplateGUI($this);
                $this->ctrl->forwardCommand($wptgui);
                break;

            case 'ilobjectmetadatagui':
                $this->checkPermission("write");
                $this->addHeaderAction();
                $ilTabs->activateTab("advmd");
                $md_gui = new ilObjectMetaDataGUI($this->object, "wpg");
                $this->ctrl->forwardCommand($md_gui);
                break;

            case 'ilrepositoryobjectsearchgui':
                $this->checkPermission("read");
                $this->addHeaderAction();
                $this->setSideBlock();
                $ilTabs->setTabActive("wiki_search_results");
                $ilCtrl->setReturn($this, 'view');
                $ilCtrl->setParameterByClass(ilWikiPageGUI::class, "wpg_id", null);
                $ilCtrl->setParameterByClass(ilObjWikiGUI::class, "wpg_id", null);
                $search_gui = new ilRepositoryObjectSearchGUI(
                    $this->object->getRefId(),
                    $this,
                    'view'
                );
                $ilCtrl->forwardCommand($search_gui);
                break;

            case 'ilobjnotificationsettingsgui':
                $this->checkPermission("write");
                $this->addHeaderAction();
                $ilTabs->activateTab("settings");
                $this->setSettingsSubTabs("notifications");
                $gui = new ilObjNotificationSettingsGUI($this->object->getRefId());
                $this->ctrl->forwardCommand($gui);
                break;

            case 'illtiproviderobjectsettinggui':
                $this->checkPermission("write");
                $this->addHeaderAction();
                $ilTabs->activateTab("settings");
                $this->setSettingsSubTabs("lti_provider");
                $lti_gui = new ilLTIProviderObjectSettingGUI($this->object->getRefId());
                $lti_gui->setCustomRolesForSelection($GLOBALS['DIC']->rbac()->review()->getLocalRoles($this->object->getRefId()));
                $lti_gui->offerLTIRolesForSelection(false);
                $this->ctrl->forwardCommand($lti_gui);
                break;

            case strtolower(SettingsGUI::class):
                $this->checkPermission("write");
                $this->addHeaderAction();
                $ilTabs->activateTab("settings");
                $this->setSettingsSubTabs("general_settings");
                $this->getTabs();
                $gui = $this->gui->settings()->settingsGUI(
                    $this->object->getId(),
                    $this->object->getRefId()
                );
                $this->ctrl->forwardCommand($gui);
                break;

            default:
                $this->addHeaderAction();
                if (!$cmd) {
                    $cmd = "infoScreen";
                }
                $cmd .= "Object";
                if ($cmd !== "cancelObject") {
                    if ($cmd !== "infoScreenObject") {
                        if (!in_array($cmd, array("createObject", "saveObject", "routeImportCmdObject"))) {
                            $this->checkPermission("read");
                        }
                    } else {
                        $this->checkPermission("visible");
                    }
                }
                $this->$cmd();
                break;
        }
    }

    public function viewObject(): void
    {
        $this->checkPermission("read");
        $this->gotoStartPageObject();
    }

    protected function initCreateForm(string $new_type): StandardForm
    {
        $f = $this->gui->ui()->factory();
        $lng = $this->lng;
        $form = parent::initCreateForm($new_type);
        $inputs = $form->getInputs();
        $inputs["start_page"] = $f->input()->field()->text(
            $lng->txt("wiki_start_page")
        )->withRequired(true);
        return $this->ui_factory->input()->container()->form()->standard(
            $this->ctrl->getFormAction($this, 'save'),
            $inputs
        )->withSubmitLabel($this->lng->txt($new_type . '_add'));
    }

    protected function afterSave(ilObject $new_object): void
    {
        $form = $this
            ->initCreateForm($this->requested_new_type)
            ->withRequest($this->request);
        $data = $form->getData();
        $new_object->setStartPage($data["start_page"]);
        $new_object->update();

        // always send a message
        $this->tpl->setOnScreenMessage('success', $this->lng->txt("object_added"), true);
        $this->ctrl->setParameterByClass(
            static::class,
            "ref_id",
            $new_object->getRefId()
        );
        $this->ctrl->redirectByClass([
            ilWikiHandlerGUI::class,
            ilObjWikiGUI::class,
            SettingsGUI::class
        ]);
    }

    /**
     * this one is called from the info button in the repository
     * @throws ilObjectException
     * @throws ilPermissionException
     */
    public function infoScreenObject(): void
    {
        $this->checkPermission("visible");
        $this->ctrl->redirectByClass(ilInfoScreenGUI::class, "showSummary");
    }

    public function infoScreen(): void
    {
        $ilAccess = $this->access;
        $ilUser = $this->user;
        $ilTabs = $this->tabs;
        $lng = $this->lng;

        $ilTabs->activateTab("info_short");

        if (!$ilAccess->checkAccess("visible", "", $this->object->getRefId())) {
            throw new ilPermissionException($this->lng->txt("permission_denied"));
        }

        $info = new ilInfoScreenGUI($this);
        $info->enablePrivateNotes();
        if (trim($this->object->getIntroduction()) !== "") {
            $info->addSection($lng->txt("wiki_introduction"));
            $info->addProperty("", nl2br($this->object->getIntroduction()));
        }

        // feedback from tutor; mark, status, comment
        $lpcomment = ilLPMarks::_lookupComment($ilUser->getId(), $this->object->getId());
        $mark = ilLPMarks::_lookupMark($ilUser->getId(), $this->object->getId());
        $status = ilWikiContributor::_lookupStatus($this->object->getId(), $ilUser->getId());
        if ($lpcomment !== "" || $mark !== "" || (int) $status !== ilWikiContributor::STATUS_NOT_GRADED) {
            $info->addSection($this->lng->txt("wiki_feedback_from_tutor"));
            if ($lpcomment !== "") {
                $info->addProperty(
                    $this->lng->txt("wiki_comment"),
                    $lpcomment
                );
            }
            if ($mark !== "") {
                $info->addProperty(
                    $this->lng->txt("wiki_mark"),
                    $mark
                );
            }

            if ((int) $status === ilWikiContributor::STATUS_PASSED) {
                $info->addProperty(
                    $this->lng->txt("status"),
                    $this->lng->txt("wiki_passed")
                );
            }
            if ((int) $status === ilWikiContributor::STATUS_FAILED) {
                $info->addProperty(
                    $this->lng->txt("status"),
                    $this->lng->txt("wiki_failed")
                );
            }
        }

        if ($ilAccess->checkAccess("read", "", $this->object->getRefId())) {
            $info->addButton($lng->txt("wiki_start_page"), self::getGotoLink($this->object->getRefId()));
        }

        // general information
        $this->lng->loadLanguageModule("meta");
        $this->lng->loadLanguageModule("wiki");

        // forward the command
        $this->ctrl->forwardCommand($info);
    }

    public function gotoStartPageObject(): void
    {
        ilUtil::redirect(self::getGotoLink($this->object->getRefId()));
    }

    public function addPageTabs(): void
    {
        $ilTabs = $this->tabs;
        $ilCtrl = $this->ctrl;

        $ilCtrl->setParameter(
            $this,
            "wpg_id",
            ilWikiPage::getPageIdForTitle($this->object->getId(), ilWikiUtil::makeDbTitle($this->requested_page))
        );
        $ilCtrl->setParameter($this, "page", ilWikiUtil::makeUrlTitle($this->requested_page));
        $ilTabs->addTarget(
            "wiki_what_links_here",
            $this->ctrl->getLinkTargetByClass(
                "ilwikipagegui",
                "whatLinksHere"
            ),
            "whatLinksHere"
        );
        $ilTabs->addTarget(
            "wiki_print_view",
            $this->ctrl->getLinkTargetByClass(
                "ilwikipagegui",
                "printViewSelection"
            ),
            "printViewSelection"
        );
    }

    public function addPagesSubTabs(): void
    {
        $ilTabs = $this->tabs;
        $ilCtrl = $this->ctrl;

        $ilTabs->activateTab("wiki_pages");

        $ilCtrl->setParameter(
            $this,
            "wpg_id",
            ilWikiPage::getPageIdForTitle(
                $this->object->getId(),
                ilWikiUtil::makeDbTitle($this->requested_page)
            )
        );
        $ilCtrl->setParameter($this, "page", ilWikiUtil::makeUrlTitle($this->requested_page));
        $ilTabs->addSubTabTarget(
            "wiki_all_pages",
            $this->ctrl->getLinkTarget($this, "allPages"),
            "allPages"
        );
        $ilTabs->addSubTabTarget(
            "wiki_recent_changes",
            $this->ctrl->getLinkTarget($this, "recentChanges"),
            "recentChanges"
        );
        $ilTabs->addSubTabTarget(
            "wiki_new_pages",
            $this->ctrl->getLinkTarget($this, "newPages"),
            "newPages"
        );
        $ilTabs->addSubTabTarget(
            "wiki_popular_pages",
            $this->ctrl->getLinkTarget($this, "popularPages"),
            "popularPages"
        );
        $ilTabs->addSubTabTarget(
            "wiki_orphaned_pages",
            $this->ctrl->getLinkTarget($this, "orphanedPages"),
            "orphanedPages"
        );
    }

    protected function getTabs(): void
    {
        $ilCtrl = $this->ctrl;
        $ilAccess = $this->access;
        $lng = $this->lng;
        $ilHelp = $this->help;

        $ilHelp->setScreenIdComponent("wiki");


        // wiki tabs
        if (in_array(strtolower($ilCtrl->getNextClass($this)), [strtolower(SettingsGUI::class)]) ||
            in_array(
                strtolower($ilCtrl->getCmdClass()),
                array("", "ilobjectcontentstylesettingsgui", "ilobjwikigui",
            "ilinfoscreengui", "ilpermissiongui", "ilexportgui", "ilratingcategorygui", "ilobjnotificationsettingsgui", "iltaxmdgui",
            "ilwikistatgui", "ilwikipagetemplategui", "iladvancedmdsettingsgui", "ilsettingspermissiongui", 'ilrepositoryobjectsearchgui',
            'ilobjecttranslationgui')
            ) || $ilCtrl->getNextClass() === "ilpermissiongui") {
            if ($this->requested_page !== "") {
                $page_id = ($this->edit_request->getWikiPageId() > 0)
                    ? $this->edit_request->getWikiPageId()
                    : $this->pm->getPageIdForTitle(
                        $this->requested_page,
                        $this->edit_request->getTranslation()
                    );
                if (is_null($page_id) && $this->edit_request->getFromPage() !== "") {
                    $page_id = $this->pm->getPageIdForTitle(
                        $this->edit_request->getFromPage(),
                        $this->edit_request->getTranslation()
                    );
                }
                if (!is_null($page_id)) {
                    $this->tabs_gui->setBackTarget(
                        $lng->txt("wiki_last_visited_page"),
                        $this->pm->getPermaLink(
                            $page_id,
                            $this->edit_request->getTranslation()
                        )
                    );
                }
            }

            // pages
            if ($ilAccess->checkAccess('read', "", $this->object->getRefId())) {
                $this->tabs_gui->addTab(
                    "wiki_pages",
                    $lng->txt("wiki_pages"),
                    $this->ctrl->getLinkTarget($this, "allPages")
                );
            }

            // info screen
            if ($ilAccess->checkAccess('visible', "", $this->object->getRefId())) {
                $this->tabs_gui->addTab(
                    "info_short",
                    $lng->txt("info_short"),
                    $this->ctrl->getLinkTargetByClass("ilinfoscreengui", "showSummary")
                );
            }

            // settings
            if ($ilAccess->checkAccess('write', "", $this->object->getRefId())) {
                $this->tabs_gui->addTab(
                    "settings",
                    $lng->txt("settings"),
                    $this->ctrl->getLinkTargetByClass(SettingsGUI::class)
                );

                // metadata
                $mdgui = new ilObjectMetaDataGUI($this->object, "wpg");
                $mdtab = $mdgui->getTab();
                if ($mdtab) {
                    $this->tabs_gui->addTab(
                        "advmd",
                        $this->lng->txt("meta_data"),
                        $mdtab
                    );
                }
            }

            // contributors
            if ($ilAccess->checkAccess('write', "", $this->object->getRefId())) {
                $this->tabs_gui->addTab(
                    "wiki_contributors",
                    $lng->txt("wiki_contributors"),
                    $this->ctrl->getLinkTarget($this, "listContributors")
                );
            }

            // statistics
            if ($ilAccess->checkAccess('statistics_read', "", $this->object->getRefId())) {
                $this->tabs_gui->addTab(
                    "statistics",
                    $lng->txt("statistics"),
                    $this->ctrl->getLinkTargetByClass("ilWikiStatGUI", "initial")
                );
            }

            if ($ilAccess->checkAccess("write", "", $this->object->getRefId())) {
                $this->tabs_gui->addTab(
                    "export",
                    $lng->txt("export"),
                    $this->ctrl->getLinkTargetByClass("ilexportgui", "")
                );
            }

            // edit permissions
            if ($ilAccess->checkAccess('edit_permission', "", $this->object->getRefId())) {
                $this->tabs_gui->addTab(
                    "perm_settings",
                    $lng->txt("perm_settings"),
                    $this->ctrl->getLinkTargetByClass("ilpermissiongui", "perm")
                );
            }
        }
    }

    public function setSettingsSubTabs(string $a_active): void
    {
        $ilTabs = $this->tabs;
        $ilCtrl = $this->ctrl;
        $lng = $this->lng;
        $ilAccess = $this->access;
        if (in_array(
            $a_active,
            array("general_settings", "style", "imp_pages", "rating_categories", "obj_multilinguality",
            "page_templates", "advmd", "permission_settings", "notifications", "lti_provider")
        )) {
            if ($ilAccess->checkAccess("write", "", $this->object->getRefId())) {
                // general properties
                $ilTabs->addSubTab(
                    "general_settings",
                    $lng->txt("wiki_general_settings"),
                    $ilCtrl->getLinkTarget($this, 'editSettings')
                );

                // permission settings
                $ilTabs->addSubTab(
                    "permission_settings",
                    $lng->txt("obj_permission_settings"),
                    $this->ctrl->getLinkTargetByClass("ilsettingspermissiongui", "")
                );

                // style properties
                $ilTabs->addSubTab(
                    "style",
                    $lng->txt("wiki_style"),
                    $ilCtrl->getLinkTargetByClass("ilObjectContentStyleSettingsGUI", "")
                );
            }

            if ($ilAccess->checkAccess("write", "", $this->object->getRefId())) {
                // important pages
                $ilTabs->addSubTab(
                    "imp_pages",
                    $lng->txt("wiki_navigation"),
                    $ilCtrl->getLinkTarget($this, 'editImportantPages')
                );
            }

            if ($ilAccess->checkAccess("write", "", $this->object->getRefId())) {
                // page templates
                $ilTabs->addSubTab(
                    "page_templates",
                    $lng->txt("wiki_page_templates"),
                    $ilCtrl->getLinkTargetByClass("ilwikipagetemplategui", "")
                );

                // rating categories
                if ($this->object->getRatingPages() && $this->object->getRatingCategories()) {
                    $lng->loadLanguageModule("rating");
                    $ilTabs->addSubTab(
                        "rating_categories",
                        $lng->txt("rating_categories"),
                        $ilCtrl->getLinkTargetByClass(array('ilratinggui', 'ilratingcategorygui'), '')
                    );
                }

                $ilTabs->addSubTab(
                    'notifications',
                    $lng->txt("notifications"),
                    $ilCtrl->getLinkTargetByClass("ilobjnotificationsettingsgui", '')
                );

                $ilTabs->addSubTab(
                    'obj_multilinguality',
                    $lng->txt("obj_multilinguality"),
                    $this->ctrl->getLinkTargetByClass("ilobjecttranslationgui", "")
                );
            }

            // LTI Provider
            $lti_settings = new ilLTIProviderObjectSettingGUI($this->object->getRefId());
            if ($lti_settings->hasSettingsAccess()) {
                $ilTabs->addSubTabTarget(
                    'lti_provider',
                    $this->ctrl->getLinkTargetByClass(ilLTIProviderObjectSettingGUI::class)
                );
            }

            $ilTabs->activateSubTab($a_active);
        }
    }

    public function editSettingsObject(): void
    {
        $this->ctrl->redirectByClass(SettingsGUI::class);
    }

    public function listContributorsObject(): void
    {
        $tpl = $this->tpl;
        $ilTabs = $this->tabs;

        $this->checkPermission("write");
        $ilTabs->activateTab("wiki_contributors");

        $table_gui = new ilWikiContributorsTableGUI(
            $this,
            "listContributors",
            $this->object->getId()
        );

        $tpl->setContent($table_gui->getHTML());

        $this->setSideBlock();
    }

    public function saveGradingObject(): void
    {
        $ilCtrl = $this->ctrl;
        $lng = $this->lng;

        $this->checkPermission("write");

        $users = $this->edit_request->getUserIds();
        $marks = $this->edit_request->getMarks();
        $comments = $this->edit_request->getComments();
        $status = $this->edit_request->getStatus();

        $saved = false;
        foreach ($users as $user_id) {
            if ($user_id != "") {
                $marks_obj = new ilLPMarks($this->object->getId(), $user_id);
                $new_mark = ilUtil::stripSlashes($marks[$user_id]);
                $new_comment = ilUtil::stripSlashes($comments[$user_id] ?? "");
                $new_status = ilUtil::stripSlashes($status[$user_id]);

                if ($marks_obj->getMark() !== $new_mark ||
                    $marks_obj->getComment() !== $new_comment ||
                    (int) ilWikiContributor::_lookupStatus($this->object->getId(), $user_id) !== (int) $new_status) {
                    ilWikiContributor::_writeStatus($this->object->getId(), $user_id, $new_status);
                    $marks_obj->setMark($new_mark);
                    $marks_obj->setComment($new_comment);
                    $marks_obj->update();
                    $saved = true;
                }
            }
        }
        if ($saved) {
            $this->tpl->setOnScreenMessage('success', $lng->txt("msg_obj_modified"), true);
        }

        $ilCtrl->redirect($this, "listContributors");
    }

    // add wiki to locator
    protected function addLocatorItems(): void
    {
        $ilLocator = $this->locator;

        if (is_object($this->object)) {
            $ilLocator->addItem(
                $this->object->getTitle(),
                self::getGotoLink($this->object->getRefId()),
                "",
                $this->requested_ref_id
            );
        }
    }

    public static function _goto(string $a_target): void
    {
        global $DIC;
        $main_tpl = $DIC->ui()->mainTemplate();

        $wpg_id = 0;
        $lang = "";
        $ilAccess = $DIC->access();
        $lng = $DIC->language();
        $ctrl = $DIC->ctrl();
        $transl = $DIC->wiki()->internal()->gui()->request()->getTranslation();
        $DIC->ctrl()->setParameterByClass(self::class, "transl", $transl);


        $i = strpos($a_target, "_");
        $a_page = "";
        if ($i > 0) {
            $a_page = substr($a_target, $i + 1);
            $a_target = substr($a_target, 0, $i);
        }

        if ($a_target === "wpage") {
            $a_page_arr = explode("_", $a_page);
            $wpg_id = (int) $a_page_arr[0];
            $ref_id = (int) ($a_page_arr[1] ?? 0);
            $lang = ($a_page_arr[2] ?? "");
            $w_id = ilWikiPage::lookupWikiId($wpg_id);
            if ($ref_id > 0) {
                $refs = array($ref_id);
            } else {
                $refs = ilObject::_getAllReferences($w_id);
            }
            foreach ($refs as $r) {
                if ($ilAccess->checkAccess("read", "", $r)) {
                    $a_target = $r;
                    $a_page = ilWikiPage::lookupTitle($wpg_id);
                }
            }
        }

        if ($ilAccess->checkAccess("read", "", $a_target)) {
            if ($wpg_id > 0) {
                $ctrl->setParameterByClass(
                    "ilobjwikigui",
                    "wpg_id",
                    $wpg_id
                );
            } else {
                $ctrl->setParameterByClass(
                    "ilobjwikigui",
                    "page",
                    ilWikiUtil::makeUrlTitle($a_page)
                );
            }
            if ($lang != "") {
                $ctrl->setParameterByClass(
                    "ilobjwikigui",
                    "transl",
                    $lang
                );
            }
            $ctrl->setParameterByClass(
                "ilwikihandlergui",
                "ref_id",
                $a_target
            );
            if ($a_page != "") {
                $ctrl->redirectByClass(
                    ["ilwikihandlergui", "ilobjwikigui"],
                    "viewPage"
                );
            } else {
                $ctrl->redirectByClass(
                    ["ilwikihandlergui"],
                    "view"
                );
            }
        } elseif ($ilAccess->checkAccess("visible", "", $a_target)) {
            ilObjectGUI::_gotoRepositoryNode($a_target, "infoScreen");
        } elseif ($ilAccess->checkAccess("read", "", ROOT_FOLDER_ID)) {
            $main_tpl->setOnScreenMessage('failure', sprintf(
                $lng->txt("msg_no_perm_read_item"),
                ilObject::_lookupTitle(ilObject::_lookupObjId($a_target))
            ), true);
            ilObjectGUI::_gotoRepositoryRoot();
        }

        throw new ilPermissionException($lng->txt("permission_denied"));
    }

    public static function getGotoLink(
        int $a_ref_id,
        string $a_page = "",
        string $lang = "-"
    ): string {
        if ($a_page === "") {
            $a_page = ilObjWiki::_lookupStartPage(ilObject::_lookupObjId($a_ref_id));
        }

        $append = (!in_array($lang, ["", "-"]))
            ? "&transl=" . $lang
            : "";
        $goto = "goto.php?target=wiki_" . $a_ref_id . "_" .
            ilWikiUtil::makeUrlTitle($a_page) . $append;
        return $goto;
    }

    public function viewPageObject(): void
    {
        $lng = $this->lng;
        $ilCtrl = $this->ctrl;
        $tpl = $this->tpl;
        $ilTabs = $this->tabs;
        $ilAccess = $this->access;

        $this->checkPermission("read");

        $ilTabs->clearTargets();
        $tpl->setHeaderActionMenu("");

        $wpage_gui = $this->gui->content()->getCurrentPageGUI();

        // page exists, show it !
        $ilCtrl->setParameter(
            $this,
            "page",
            ilWikiUtil::makeUrlTitle($wpage_gui->getPageObject()->getTitle())
        );

        $wpage_gui->setStyleId($this->content_style_domain->getEffectiveStyleId());

        $this->setContentStyleSheet();

        //$wpage_gui->setSideBlock();
        $ilCtrl->redirectByClass(ilWikiPageGUI::class, "preview");
    }

    public function allPagesObject(): void
    {
        $tpl = $this->tpl;

        $this->checkPermission("read");

        $this->addPagesSubTabs();

        $table_gui = new ilWikiPagesTableGUI(
            $this,
            "allPages",
            $this->object->getId(),
            IL_WIKI_ALL_PAGES
        );

        //$this->setSideBlock();
        $tpl->setContent($table_gui->getHTML());
    }

    /**
     * @throws ilObjectException
     */
    public function popularPagesObject(): void
    {
        $tpl = $this->tpl;

        $this->checkPermission("read");

        $this->addPagesSubTabs();

        $table_gui = new ilWikiPagesTableGUI(
            $this,
            "popularPages",
            $this->object->getId(),
            IL_WIKI_POPULAR_PAGES
        );

        //$this->setSideBlock();
        $tpl->setContent($table_gui->getHTML());
    }

    /**
     * @throws ilObjectException
     */
    public function orphanedPagesObject(): void
    {
        $tpl = $this->tpl;

        $this->checkPermission("read");

        $this->addPagesSubTabs();

        $table_gui = new ilWikiPagesTableGUI(
            $this,
            "orphanedPages",
            $this->object->getId(),
            IL_WIKI_ORPHANED_PAGES
        );

        //$this->setSideBlock();
        $tpl->setContent($table_gui->getHTML());
    }

    /**
     * @param string $a_page page title
     */
    public function gotoPageObject(
        string $a_page = ""
    ): void {
        $ilCtrl = $this->ctrl;

        if ($a_page === "") {
            $a_page = $this->requested_page;
        }

        if ($this->pm->existsByTitle(
            ilWikiUtil::makeDbTitle($a_page),
            $this->edit_request->getTranslation()
        )) {
            // to do: get rid of this redirect
            ilUtil::redirect(self::getGotoLink($this->object->getRefId(), $a_page, $this->edit_request->getTranslation()));
        } else {
            if (!$this->access->checkAccess("add_pages", "", $this->object->getRefId())) {
                $this->tpl->setOnScreenMessage("failure", $this->lng->txt("no_permission"), true);
                $from_page_id = $this->pm->getPageIdForTitle($this->edit_request->getFromPage(), $this->edit_request->getTranslation());
                ilUtil::redirect($this->pm->getPermaLink($from_page_id, $this->edit_request->getTranslation()));
            }
            if ($this->isNewTranslatedPage()) {
                return;
            }
            if (!$this->object->getTemplateSelectionOnCreation()) {
                // check length
                if (ilStr::strLen(ilWikiUtil::makeDbTitle($a_page)) > 200) {
                    $this->tpl->setOnScreenMessage(
                        'failure',
                        $this->lng->txt("wiki_page_title_too_long") . " (" . $a_page . ")",
                        true
                    );
                    $ilCtrl->setParameterByClass(
                        "ilwikipagegui",
                        "page",
                        ilWikiUtil::makeUrlTitle($this->edit_request->getFromPage())
                    );
                    $ilCtrl->redirectByClass("ilwikipagegui", "preview");
                }

                $this->pm->createWikiPage(
                    $a_page,
                    0,
                    $this->edit_request->getTranslation()
                );

                // redirect to newly created page
                $ilCtrl->setParameterByClass("ilwikipagegui", "page", ilWikiUtil::makeUrlTitle(($a_page)));
                $ilCtrl->redirectByClass("ilwikipagegui", "edit");
            } else {
                $ilCtrl->setParameter($this, "page", ilWikiUtil::makeUrlTitle($this->requested_page));
                $ilCtrl->setParameter(
                    $this,
                    "from_page",
                    ilWikiUtil::makeUrlTitle($this->edit_request->getFromPage())
                );
                $ilCtrl->redirect($this, "showTemplateSelection");
            }
        }
    }

    protected function isNewTranslatedPage(): bool
    {
        if (in_array($this->edit_request->getTranslation(), ["-", ""])) {
            return false;
        }
        $page = $this->requested_page;
        if (!ilWikiPage::_wikiPageExists(
            $this->object->getId(),
            ilWikiUtil::makeDbTitle($page),
            $this->edit_request->getTranslation()
        )) {
            $this->tabs_gui->clearTargets();
            $this->tabs_gui->setBackTarget(
                $this->lng->txt("back"),
                $this->pm->getPermaLink(
                    $this->pm->getPageIdForTitle(
                        $this->edit_request->getFromPage(),
                        $this->edit_request->getTranslation()
                    ),
                    $this->edit_request->getTranslation()
                )
            );
            $form = $this->getNewTranslatedPageForm();
            $this->tpl->setContent($this->getRenderedTranslationInfo() . $form->render());
            return true;
        }
        return false;
    }

    protected function getRenderedTranslationInfo()
    {
        $mess = $this->gui->ui()->factory()->messageBox()->info(
            $this->lng->txt("wiki_translate_page_master_info")
        );
        return $this->gui->ui()->renderer()->render($mess);
    }

    protected function getNewTranslatedPageForm(): \ILIAS\Repository\Form\FormAdapterGUI
    {
        $pm = $this->domain->page()->page($this->object->getRefId());
        $options = [];
        foreach ($pm->getMasterPagesWithoutTranslation($this->edit_request->getTranslation()) as $page) {
            $options[$page->getId()] = $page->getTitle();
        }
        $append = " '" . $this->edit_request->getPage() . "'";
        $append .= " (" . $this->lng->txt("meta_l_" . $this->edit_request->getTranslation()) . ")";
        $append2 = " (" . $this->lng->txt("meta_l_" . $this->ot->getMasterLanguage()) . ")";
        $form = $this->gui->form([self::class], "createNewTranslatedPage")
            ->section("sec", $this->lng->txt("wiki_translation_page") . $append)
            ->switch("type", $this->lng->txt("wiki_page_in_master_language") . $append2, "", "existing")
            ->group("existing", $this->lng->txt("wiki_master_existing"))
            ->select("master_id", $this->lng->txt("wiki_master_title"), $options)
            ->required()
            ->group("new", $this->lng->txt("wiki_no_master"))
            ->text("master_title", $this->lng->txt("wiki_master_title"))
            ->end();

        if ($this->object->getTemplateSelectionOnCreation($this->edit_request->getTranslation())) {
            $form = $form->radio("page_templ", $this->lng->txt("wiki_page_template"), "", "0")->required();
            if ($this->object->getEmptyPageTemplate()) {
                $form = $form->radioOption("0", $this->lng->txt("wiki_empty_page"));
            }
            $wt = new ilWikiPageTemplate($this->object->getId());
            $ts = $wt->getAllInfo(ilWikiPageTemplate::TYPE_NEW_PAGES, $this->edit_request->getTranslation());
            foreach ($ts as $t) {
                $form = $form->radioOption($t["wpage_id"], $t["title"]);
            }
        }

        return $form;
    }

    protected function createNewTranslatedPageObject(): void
    {
        $form = $this->getNewTranslatedPageForm();
        if ($form->isValid()) {
            $page_template_id = 0;
            if ($this->object->getTemplateSelectionOnCreation($this->edit_request->getTranslation())) {
                $page_template_id = (int) $form->getData("page_templ");
            }
            if ($form->getData("type") === "new") {
                $wpg_id = $this->pm->createWikiPage(
                    $form->getData("master_title")
                );
            } else {
                $wpg_id = $form->getData("master_id");
            }
            $wpg_id = $this->pm->createWikiPage(
                $this->edit_request->getPage(),
                $wpg_id,
                $this->edit_request->getTranslation(),
                $page_template_id
            );
        }
        $this->ctrl->setParameterByClass(ilWikiPageGUI::class, "wpg_id", $wpg_id);
        $this->ctrl->redirectByClass(ilWikiPageGUI::class, "preview");
    }

    public function randomPageObject(): void
    {
        $this->checkPermission("read");

        $page = ilWikiPage::getRandomPage($this->object->getId());
        $this->gotoPageObject($page);
    }

    public function recentChangesObject(): void
    {
        $tpl = $this->tpl;

        $this->checkPermission("read");

        $this->addPagesSubTabs();

        $table_gui = new ilWikiRecentChangesTableGUI(
            $this,
            "recentChanges",
            $this->object->getId()
        );

        //$this->setSideBlock();
        $tpl->setContent($table_gui->getHTML());
    }

    public function setSideBlock(int $a_wpg_id = 0): void
    {
        self::renderSideBlock($a_wpg_id, $this->object->getRefId());
    }

    public static function renderSideBlock(
        int $a_wpg_id,
        int $a_wiki_ref_id,
        ?ilWikiPage $a_wp = null
    ): void {
        global $DIC;

        $tpl = $DIC["tpl"];
        $lng = $DIC->language();
        $ilAccess = $DIC->access();
        $ilCtrl = $DIC->ctrl();

        $debug = false;
        if ($debug) {
            $tpl->addJavaScript("../components/ILIAS/Wiki/resources/WikiPres.js");
        } else {
            $tpl->addJavaScript("assets/js/WikiPres.js");
        }

        // setting asynch to false fixes #0019457, since otherwise ilBlockGUI would act on asynch and output html when side blocks
        // being processed during the export. This is a flaw in ilCtrl and/or ilBlockGUI.
        $tpl->addOnLoadCode("il.Wiki.Pres.init('" . $ilCtrl->getLinkTargetByClass("ilobjwikigui", "", "", false, false) . "');");

        if ($a_wpg_id > 0 && !$a_wp) {
            $a_wp = new ilWikiPage($a_wpg_id);
        }

        // search block
        $rcontent = ilRepositoryObjectSearchGUI::getSearchBlockHTML($lng->txt('wiki_search'));


        // quick navigation
        if ($a_wpg_id > 0) {
            // rating
            $wiki_id = ilObject::_lookupObjId($a_wiki_ref_id);
            if (ilObjWiki::_lookupRating($wiki_id) &&
                // ilObjWiki::_lookupRatingAsBlock($wiki_id) &&
                $a_wp->getRating()) {
                $rgui = new ilRatingGUI();
                $rgui->setObject($wiki_id, "wiki", $a_wpg_id, "wpg");
                $rgui->enableCategories(ilObjWiki::_lookupRatingCategories($wiki_id));
                $rgui->setYourRatingText("#");
                $rcontent .= $rgui->getBlockHTML($lng->txt("wiki_rate_page"));
            }

            // advanced metadata
            if (!ilWikiPage::lookupAdvancedMetadataHidden($a_wpg_id)) {
                $cmd = null;
                if ($ilAccess->checkAccess("write", "", $a_wiki_ref_id) ||
                    $ilAccess->checkAccess("edit_page_meta", "", $a_wiki_ref_id)) {
                    $cmd = array(
                        "edit" => $ilCtrl->getLinkTargetByClass("ilwikipagegui", "editAdvancedMetaData"),
                        "hide" => $ilCtrl->getLinkTargetByClass("ilwikipagegui", "hideAdvancedMetaData")
                    );
                }
                $wiki = new ilObjWiki($a_wiki_ref_id);
                $callback = $wiki->getLinkMetadataValues()
                    ? array($wiki, "decorateAdvMDValue")
                    : null;
                $mdgui = new ilObjectMetaDataGUI($wiki, "wpg", $a_wpg_id);
                $rcontent .= $mdgui->getBlockHTML($cmd, $callback); // #17291
            }
        }

        // important pages
        $imp_pages_block = new ilWikiImportantPagesBlockGUI();
        $rcontent .= $imp_pages_block->getHTML();

        // wiki functions block
        if ($a_wpg_id > 0) {
            $wiki_functions_block = new ilWikiFunctionsBlockGUI();
            $wiki_functions_block->setPageObject($a_wp);
            $rcontent .= $wiki_functions_block->getHTML();
        }

        $tpl->setRightContent($rcontent);
    }

    public function newPagesObject(): void
    {
        $tpl = $this->tpl;

        $this->checkPermission("read");

        $this->addPagesSubTabs();

        $table_gui = new ilWikiPagesTableGUI(
            $this,
            "newPages",
            $this->object->getId(),
            IL_WIKI_NEW_PAGES
        );

        //$this->setSideBlock();
        $tpl->setContent($table_gui->getHTML());
    }

    protected function getPrintPageIds(): array
    {
        $page_ids = [];
        $ordering = $this->edit_request->getPrintOrdering();

        // multiple ordered page ids
        if (count($ordering) > 0) {
            asort($ordering);
            $page_ids = array_keys($ordering);
        }
        // single page
        elseif ($this->edit_request->getWikiPageId()) {
            $page_ids = array($this->edit_request->getWikiPageId());
        }
        return $page_ids;
    }

    public function getPrintView(bool $export = false): \ILIAS\Export\PrintProcessGUI
    {
        $page_ids = $export
            ? null
            : $this->getPrintPageIds();
        $provider = new \ILIAS\Wiki\WikiPrintViewProviderGUI(
            $this->lng,
            $this->ctrl,
            $this->object->getRefId(),
            $page_ids
        );

        return new \ILIAS\Export\PrintProcessGUI(
            $provider,
            $this->http,
            $this->ui,
            $this->lng
        );
    }

    public function printViewObject(): void
    {
        $print_view = $this->getPrintView();
        $print_view->sendPrintView();
    }

    public function performSearchObject(): void
    {
        $tpl = $this->tpl;
        $ilTabs = $this->tabs;
        $ilCtrl = $this->ctrl;
        $lng = $this->lng;

        $this->checkPermission("read");

        $ilTabs->setTabActive("wiki_search_results");

        if ($this->edit_request->getSearchTerm() === "") {
            $this->tpl->setOnScreenMessage('failure', $lng->txt("wiki_please_enter_search_term"), true);
            $ilCtrl->redirectByClass("ilwikipagegui", "preview");
        }

        $search_results = ilObjWiki::_performSearch(
            $this->object->getId(),
            $this->edit_request->getSearchTerm()
        );
        $table_gui = new ilWikiSearchResultsTableGUI(
            $this,
            "performSearch",
            $this->object->getId(),
            $search_results,
            $this->edit_request->getSearchTerm()
        );

        $this->setSideBlock();
        $tpl->setContent($table_gui->getHTML());
    }

    public function setContentStyleSheet(): void
    {
        $tpl = $this->tpl;

        if ($tpl == null) {
            $tpl = $this->tpl;
        }

        $this->content_style_gui->addCss($tpl, $this->object->getRefId());
        $tpl->addCss(ilObjStyleSheet::getSyntaxStylePath());
    }


    //
    // Important pages
    //

    public function editImportantPagesObject(): void
    {
        $tpl = $this->tpl;
        $ilToolbar = $this->toolbar;
        $ilTabs = $this->tabs;
        $lng = $this->lng;
        $ilCtrl = $this->ctrl;

        $this->checkPermission("edit_wiki_navigation");

        $this->tpl->setOnScreenMessage('info', $lng->txt("wiki_navigation_info"));

        $ipages_ids = $this->imp_pages->getImportantPageIds();

        // list pages
        $pages = $this->pm->getAllPagesInfo();
        $options = array("" => $lng->txt("please_select"));
        foreach ($pages as $p) {
            if (!in_array($p->getId(), $ipages_ids)) {
                $options[$p->getId()] = ilStr::shortenTextExtended($p->getTitle(), 60, true);
            }
        }
        if (count($options) > 0) {
            $si = new ilSelectInputGUI($lng->txt("wiki_pages"), "imp_page_id");
            $si->setOptions($options);
            $si->setInfo($lng->txt(""));
            $ilToolbar->addInputItem($si);
            $ilToolbar->setFormAction($ilCtrl->getFormAction($this));
            $ilToolbar->addFormButton($lng->txt("add"), "addImportantPage");
        }


        $ilTabs->activateTab("settings");
        $this->setSettingsSubTabs("imp_pages");

        $imp_table = new ilImportantPagesTableGUI($this, "editImportantPages");

        $tpl->setContent($imp_table->getHTML());
    }

    public function addImportantPageObject(): void
    {
        $ilCtrl = $this->ctrl;
        $lng = $this->lng;

        $this->checkPermission("edit_wiki_navigation");

        $imp_page_id = $this->edit_request->getImportantPageId();
        if ($imp_page_id > 0) {
            $this->imp_pages->add($imp_page_id);
            $this->tpl->setOnScreenMessage('success', $lng->txt("wiki_imp_page_added"), true);
        }
        $ilCtrl->redirect($this, "editImportantPages");
    }

    public function confirmRemoveImportantPagesObject(): void
    {
        $ilCtrl = $this->ctrl;
        $tpl = $this->tpl;
        $lng = $this->lng;

        $imp_page_ids = $this->edit_request->getImportantPageIds();
        if (count($imp_page_ids) === 0) {
            $this->tpl->setOnScreenMessage('info', $lng->txt("no_checkbox"), true);
            $ilCtrl->redirect($this, "editImportantPages");
        } else {
            $cgui = new ilConfirmationGUI();
            $cgui->setFormAction($ilCtrl->getFormAction($this));
            $cgui->setHeaderText($lng->txt("wiki_sure_remove_imp_pages"));
            $cgui->setCancel($lng->txt("cancel"), "editImportantPages");
            $cgui->setConfirm($lng->txt("remove"), "removeImportantPages");

            foreach ($imp_page_ids as $i) {
                $cgui->addItem("imp_page_id[]", $i, ilWikiPage::lookupTitle((int) $i));
            }

            $tpl->setContent($cgui->getHTML());
        }
    }

    public function removeImportantPagesObject(): void
    {
        $ilCtrl = $this->ctrl;
        $lng = $this->lng;

        $this->checkPermission("edit_wiki_navigation");

        $imp_page_ids = $this->edit_request->getImportantPageIds();
        foreach ($imp_page_ids as $i) {
            $this->imp_pages->removeImportantPage($i);
        }
        $this->tpl->setOnScreenMessage('success', $lng->txt("wiki_removed_imp_pages"), true);
        $ilCtrl->redirect($this, "editImportantPages");
    }

    public function saveOrderingAndIndentObject(): void
    {
        $ilCtrl = $this->ctrl;
        $lng = $this->lng;

        $this->checkPermission("edit_wiki_navigation");

        $ordering = $this->edit_request->getImportantPageOrdering();
        $indentation = $this->edit_request->getImportantPageIndentation();
        $this->imp_pages->saveOrderingAndIndentation($ordering, $indentation);
        $this->tpl->setOnScreenMessage('success', $lng->txt("wiki_ordering_and_indent_saved"), true);
        $ilCtrl->redirect($this, "editImportantPages");
    }

    public function setAsStartPageObject(): void
    {
        $ilCtrl = $this->ctrl;
        $lng = $this->lng;

        $this->checkPermission("edit_wiki_navigation");

        $imp_page_ids = $this->edit_request->getImportantPageIds();
        if (count($imp_page_ids) !== 1) {
            $this->tpl->setOnScreenMessage('info', $lng->txt("wiki_select_one_item"), true);
        } else {
            $this->imp_pages->removeImportantPage($imp_page_ids[0]);
            $this->object->setStartPage(ilWikiPage::lookupTitle($imp_page_ids[0]));
            $this->object->update();
            $this->tpl->setOnScreenMessage('success', $this->lng->txt("msg_obj_modified"), true);
        }
        $ilCtrl->redirect($this, "editImportantPages");
    }


    /**
     * Create html package
     * @throws ilTemplateException
     * @throws ilWikiExportException
     */
    public function exportHTML(): void
    {
        /** @var ilObjWiki $wiki */
        $wiki = $this->object;
        $cont_exp = new Export\WikiHtmlExport($wiki);

        $format = explode("_", $this->edit_request->getFormat());
        if (($format[1] ?? "") === "comments") {
            $cont_exp->setMode(Export\WikiHtmlExport::MODE_COMMENTS);
        }

        $cont_exp->buildExportFile();
    }

    /**
     * Get title for wiki page (used in ilNotesGUI)
     */
    public static function lookupSubObjectTitle(
        int $a_wiki_id,
        string $a_page_id
    ): string {
        $page = new ilWikiPage($a_page_id);
        if ($page->getWikiId() === $a_wiki_id) {
            return $page->getTitle();
        }
        return "";
    }

    /**
     * Used for rating export
     */
    public function getSubObjectTitle(
        int $a_id,
        string $a_type
    ): string {
        return ilWikiPage::lookupTitle($a_id);
    }

    public function showTemplateSelectionObject(): void
    {
        $lng = $this->lng;
        $tpl = $this->tpl;
        $ilTabs = $this->tabs;
        $ilCtrl = $this->ctrl;

        $ilCtrl->setParameterByClass(
            "ilobjwikigui",
            "from_page",
            ilWikiUtil::makeUrlTitle($this->edit_request->getFromPage())
        );
        $ilTabs->clearTargets();
        $this->tpl->setOnScreenMessage('info', $lng->txt("wiki_page_not_exist_select_templ"));

        $form = $this->initTemplateSelectionForm();
        $tpl->setContent($form->getHTML());
    }

    public function initTemplateSelectionForm(): ilPropertyFormGUI
    {
        $lng = $this->lng;
        $ilCtrl = $this->ctrl;

        $form = new ilPropertyFormGUI();

        // page name
        $hi = new ilHiddenInputGUI("page");
        $hi->setValue($this->requested_page);
        $form->addItem($hi);

        // page template
        $radg = new ilRadioGroupInputGUI($lng->txt("wiki_page_template"), "page_templ");
        $radg->setRequired(true);

        if ($this->object->getEmptyPageTemplate()) {
            $op1 = new ilRadioOption($lng->txt("wiki_empty_page"), 0);
            $radg->addOption($op1);
        }

        $wt = new ilWikiPageTemplate($this->object->getId());
        $ts = $wt->getAllInfo(ilWikiPageTemplate::TYPE_NEW_PAGES, $this->edit_request->getTranslation());
        foreach ($ts as $t) {
            $op = new ilRadioOption($t["title"], $t["wpage_id"]);
            $radg->addOption($op);
        }

        $form->addItem($radg);

        // save and cancel commands
        $form->addCommandButton("createPageUsingTemplate", $lng->txt("wiki_create_page"));
        $form->addCommandButton("cancelCreationPageUsingTemplate", $lng->txt("cancel"));

        $form->setTitle($lng->txt("wiki_new_page") . ": " . $this->requested_page);
        $form->setFormAction($ilCtrl->getFormAction($this));

        return $form;
    }

    public function createPageUsingTemplateObject(): void
    {
        $tpl = $this->tpl;
        $lng = $this->lng;
        $ilCtrl = $this->ctrl;

        $form = $this->initTemplateSelectionForm();
        if ($form->checkInput()) {
            $a_page = $this->edit_request->getPage();
            $this->pm->createWikiPage(
                $a_page,
                0,
                $this->edit_request->getTranslation(),
                $this->edit_request->getPageTemplateId()
            );

            // redirect to newly created page
            $ilCtrl->setParameterByClass("ilwikipagegui", "page", ilWikiUtil::makeUrlTitle(($a_page)));
            $ilCtrl->redirectByClass("ilwikipagegui", "edit");

            $this->tpl->setOnScreenMessage('success', $lng->txt("msg_obj_modified"), true);
            $ilCtrl->redirect($this, "");
        } else {
            $form->setValuesByPost();
            $tpl->setContent($form->getHTML());
        }
    }

    public function cancelCreationPageUsingTemplateObject(): void
    {
        $ilCtrl = $this->ctrl;

        // redirect to newly created page
        $ilCtrl->setParameterByClass(
            "ilwikipagegui",
            "page",
            ilWikiUtil::makeUrlTitle($this->edit_request->getFromPage())
        );
        $ilCtrl->redirectByClass("ilwikipagegui", "preview");
    }

    protected function checkPermissionBool(string $perm, string $cmd = "", string $type = "", ?int $ref_id = null): bool
    {
        if ($perm === "create") {
            return parent::checkPermissionBool($perm, $cmd, $type, $ref_id);
        } else {
            if (!$ref_id) {
                $ref_id = $this->object->getRefId();
            }
            return ilWikiPerm::check($perm, $ref_id, $cmd);
        }
    }


    //
    // User HTML Export
    //
    /**
        Current process:
        - On Button Click (Javascript): il.Wiki.Pres.startHTMLExport()
          - Ajax call to (PHP): ilObjWikiGUI->initUserHTMLExport()
            - On Ajax Success (Javascript):
              - Ajax call to (PHP): ilObjWikiGUI->startUserHTMLExport()
              - Call to il.Wiki.Pres.updateProgress
                - Ajax call to (PHP): ilObjWikiGUI->getUserHTMLExportProgress()
                  - On Ajax Success:
                    - If finished window.location.href to ilObjWikiGUI->downloadUserHTMLExport()
                    - If not finished: Wait for a second and call to il.Wiki.Pres.updateProgress
     */

    /**
     * Export html (as user)
     */
    public function initUserHTMLExportObject(): void
    {
        $this->log->debug("init: " . $this->req_with_comments);
        $this->checkPermission("wiki_html_export");
        $this->object->initUserHTMLExport($this->req_with_comments);
    }

    /**
     * Export html (as user)
     */
    public function startUserHTMLExportObject(): void
    {
        $this->log->debug("start: " . $this->req_with_comments);
        $this->checkPermission("wiki_html_export");
        $this->object->startUserHTMLExport($this->req_with_comments);
    }

    /**
     * Get user html export progress
     */
    public function getUserHTMLExportProgressObject(): void
    {
        $this->log->debug("get progress: " . $this->req_with_comments);
        $this->checkPermission("wiki_html_export");
        $p = $this->object->getUserHTMLExportProgress($this->req_with_comments);

        $pb = ilProgressBar::getInstance();
        $pb->setCurrent($p["progress"]);

        $r = new stdClass();
        $r->progressBar = $pb->render();
        $r->status = $p["status"];
        $this->log->debug("status: " . $r->status);
        echo(json_encode($r, JSON_THROW_ON_ERROR));
        exit;
    }

    public function downloadUserHTMLExportObject(): void
    {
        $this->log->debug("download");
        $this->checkPermission("wiki_html_export");
        $this->object->deliverUserHTMLExport();
    }

    public function downloadUserHTMLExportWithCommentsObject(): void
    {
        $this->log->debug("download");
        $this->checkPermission("wiki_html_export");
        $this->object->deliverUserHTMLExport(true);
    }

    protected function triggerAssignmentTool(): void
    {
        if (!is_object($this->object)) {
            return;
        }
        $ass_info = ilExcRepoObjAssignment::getInstance()->getAssignmentInfoOfObj(
            $this->object->getRefId(),
            $this->user->getId()
        );
        if (count($ass_info) > 0) {
            $ass_ids = array_map(static function ($i): int {
                return $i->getId();
            }, $ass_info);
            $this->tool_context->current()->addAdditionalData(ilExerciseGSToolProvider::SHOW_EXC_ASSIGNMENT_INFO, true);
            $this->tool_context->current()->addAdditionalData(ilExerciseGSToolProvider::EXC_ASS_IDS, $ass_ids);
            $this->tool_context->current()->addAdditionalData(
                ilExerciseGSToolProvider::EXC_ASS_BUTTONS,
                $this->getAssignmentButtons()
            );
        }
    }

    /**
     * Get assignment buttons
     */
    protected function getAssignmentButtons(): array
    {
        $ilCtrl = $this->ctrl;
        $ui = $this->ui;
        $lng = $this->lng;

        $ass_info = ilExcRepoObjAssignment::getInstance()->getAssignmentInfoOfObj(
            $this->object->getRefId(),
            $this->user->getId()
        );
        $buttons = [];
        foreach ($ass_info as $i) {	// should be only one
            $ass = new ilExAssignment($i->getId());
            $times_up = $ass->afterDeadlineStrict();

            // submit button
            if (!$times_up) {
                $ilCtrl->setParameterByClass("ilwikipagegui", "ass", $ass->getId());
                $submit_link = $ilCtrl->getLinkTargetByClass("ilwikipagegui", "finalizeAssignment");
                $ilCtrl->setParameterByClass("ilwikipagegui", "ass", "");

                $buttons[$i->getId()][] = $ui->factory()->button()->primary($lng->txt("wiki_finalize_wiki"), $submit_link);
            }

            // submitted files
            $submission = new ilExSubmission($ass, $this->user->getId());
            if ($submission->hasSubmitted()) {
                $submitted = $submission->getSelectedObject();
                if ($submitted?->getTimestamp() != "") {
                    $ilCtrl->setParameterByClass("ilwikipagegui", "ass", $ass->getId());
                }
                $dl_link = $ilCtrl->getLinkTargetByClass("ilwikipagegui", "downloadExcSubFile");
                $ilCtrl->setParameterByClass("ilwikipagegui", "ass", "");
                $buttons[$i->getId()][] = $ui->factory()->button()->standard($lng->txt("wiki_download_submission"), $dl_link);
            }
        }
        return $buttons;
    }
}
