<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

/**
 * Text question GUI representation
 *
 * The assTextQuestionGUI class encapsulates the GUI representation for text questions.
 *
 * @author	Helmut Schottmüller <helmut.schottmueller@mac.com>
 * @author	Björn Heyser <bheyser@databay.de>
 * @author	Maximilian Becker <mbecker@databay.de>
 *
 * @version	$Id$
 *
 * @ingroup components\ILIASTestQuestionPool
 * @ilCtrl_Calls assTextQuestionGUI: ilFormPropertyDispatchGUI
 */
class assTextQuestionGUI extends assQuestionGUI implements ilGuiQuestionScoringAdjustable, ilGuiAnswerScoringAdjustable
{
    protected bool $tiny_mce_enabled;
    /**
     * assTextQuestionGUI constructor
     *
     * The constructor takes possible arguments an creates an instance of the assTextQuestionGUI object.
     *
     * @param integer $id The database id of a text question object
     */
    public function __construct($id = -1)
    {
        $this->tiny_mce_enabled = (new ilSetting('advanced_editing'))->get('advanced_editing_javascript_editor')
            === 'tinymce' ? true : false;
        parent::__construct();
        $this->object = new assTextQuestion();
        if ($id >= 0) {
            $this->object->loadFromDb($id);
        }
    }

    /**
     * {@inheritdoc}
     */
    protected function writePostData(bool $always = false): int
    {
        $hasErrors = (!$always) ? $this->editQuestion(true) : false;
        if (!$hasErrors) {
            $this->writeQuestionGenericPostData();
            $this->writeQuestionSpecificPostData(new ilPropertyFormGUI());
            $this->writeAnswerSpecificPostData(new ilPropertyFormGUI());
            $this->saveTaxonomyAssignments();
            return 0;
        }
        return 1;
    }

    public function editQuestion(
        bool $checkonly = false,
        ?bool $is_save_cmd = null
    ): bool {
        $save = $is_save_cmd ?? $this->isSaveCommand();

        $form = new ilPropertyFormGUI();
        $this->editForm = $form;

        $form->setFormAction($this->ctrl->getFormAction($this));
        $form->setTitle($this->outQuestionType());
        $form->setMultipart(true);
        $form->setTableWidth("100%");
        $form->setId("asstextquestion");

        $this->addBasicQuestionFormProperties($form);
        $this->populateQuestionSpecificFormPart($form);
        $this->populateAnswerSpecificFormPart($form);


        $this->populateTaxonomyFormSection($form);

        $this->addQuestionFormCommandButtons($form);

        $errors = false;

        if ($save) {
            $form->setValuesByPost();
            $errors = !$form->checkInput();
            $form->setValuesByPost(); // again, because checkInput now performs the whole stripSlashes handling and we need this if we don't want to have duplication of backslashes
            if ($errors) {
                $checkonly = false;
            }
        }

        if (!$checkonly) {
            $this->renderEditForm($form);
        }
        return $errors;
    }

    private static function buildAnswerTextOnlyArray($answers): array
    {
        $answerTexts = [];

        foreach ($answers as $answer) {
            $answerTexts[] = $answer->getAnswertext();
        }

        return $answerTexts;
    }

    public function magicAfterTestOutput(): void
    {
        // TODO - BEGIN: what exactly is done here? cant we use the parent method?
        $rtestring = ilRTE::_getRTEClassname();
        $rte = new $rtestring();
        $rte->addUserTextEditor('textarea.textinput');

        // TODO - END: what exactly is done here? cant we use the parent method?
    }

    public function getSolutionOutput(
        int $active_id,
        ?int $pass = null,
        bool $graphical_output = false,
        bool $result_output = false,
        bool $show_question_only = true,
        bool $show_feedback = false,
        bool $show_correct_solution = false,
        bool $show_manual_scoring = false,
        bool $show_question_text = true,
        bool $show_inline_feedback = true
    ): string {
        if (($active_id > 0) && (!$show_correct_solution)) {
            $solution = $this->getUserAnswer($active_id, $pass);
        } else {
            $solution = $this->getBestAnswer($this->renderPurposeSupportsFormHtml());
        }

        return $this->renderSolutionOutput(
            $solution,
            $active_id,
            $pass,
            $graphical_output,
            $result_output,
            $show_question_only,
            $show_feedback,
            $show_correct_solution,
            $show_manual_scoring,
            $show_question_text,
            false,
            false,
        );
    }

    public function renderSolutionOutput(
        mixed $user_solutions,
        int $active_id,
        ?int $pass,
        bool $graphical_output = false,
        bool $result_output = false,
        bool $show_question_only = true,
        bool $show_feedback = false,
        bool $show_correct_solution = false,
        bool $show_manual_scoring = false,
        bool $show_question_text = true,
        bool $show_autosave_title = false,
        bool $show_inline_feedback = false,
    ): ?string {

        $template = new ilTemplate("tpl.il_as_qpl_text_question_output_solution.html", true, true, "components/ILIAS/TestQuestionPool");
        $solutiontemplate = new ilTemplate("tpl.il_as_tst_solution_output.html", true, true, "components/ILIAS/TestQuestionPool");

        $solution = $this->object->getHtmlUserSolutionPurifier()->purify($user_solutions);
        if ($this->renderPurposeSupportsFormHtml()) {
            $template->setCurrentBlock('essay_div');
            $template->setVariable("DIV_ESSAY", ilLegacyFormElementsUtil::prepareTextareaOutput($solution, true));
        } else {
            $template->setCurrentBlock('essay_textarea');
            $template->setVariable("TA_ESSAY", ilLegacyFormElementsUtil::prepareTextareaOutput($solution, true, true));
        }
        $template->parseCurrentBlock();

        if (!$show_correct_solution) {
            $max_no_of_chars = $this->object->getMaxNumOfChars();

            if ($max_no_of_chars == 0) {
                $max_no_of_chars = ucfirst($this->lng->txt('unlimited'));
            }

            $act_no_of_chars = $this->object->countLetters($solution);
            $template->setVariable("CHARACTER_INFO", '<b>' . $max_no_of_chars . '</b>' .
                $this->lng->txt('answer_characters') . ' <b>' . $act_no_of_chars . '</b>');

            if ($this->object->isWordCounterEnabled()) {
                $template->setCurrentBlock('word_count');
                $template->setVariable(
                    'WORD_COUNT',
                    $this->lng->txt('qst_essay_written_words') .
                    ' <b>' . $this->object->countWords($solution) . '</b>'
                );
                $template->parseCurrentBlock();
            }
        }
        if (($active_id > 0) && (!$show_correct_solution)) {
            if ($graphical_output) {
                $correctness_icon = $this->generateCorrectnessIconsForCorrectness(self::CORRECTNESS_NOT_OK);
                $reached_points = $this->object->getReachedPoints($active_id, $pass);
                if ($reached_points == $this->object->getMaximumPoints()) {
                    $correctness_icon = $this->generateCorrectnessIconsForCorrectness(self::CORRECTNESS_OK);
                } elseif ($reached_points > 0) {
                    $correctness_icon = $this->generateCorrectnessIconsForCorrectness(self::CORRECTNESS_MOSTLY_OK);
                }
                $template->setCurrentBlock("icon_ok");
                $template->setVariable("ICON_OK", $correctness_icon);
                $template->parseCurrentBlock();
            }
        }
        if ($show_question_text == true) {
            $template->setVariable("QUESTIONTEXT", $this->object->getQuestionForHTMLOutput());
        }
        $questionoutput = $template->get();

        $feedback = '';
        if ($show_feedback) {
            if (!$this->isTestPresentationContext()) {
                $fb = $this->getGenericFeedbackOutput((int) $active_id, $pass);
                $feedback .= strlen($fb) ? $fb : '';
            }
        }
        if (strlen($feedback)) {
            $cssClass = (
                $this->hasCorrectSolution($active_id, $pass) ?
                ilAssQuestionFeedback::CSS_CLASS_FEEDBACK_CORRECT : ilAssQuestionFeedback::CSS_CLASS_FEEDBACK_WRONG
            );

            $solutiontemplate->setVariable("ILC_FB_CSS_CLASS", $cssClass);
            $solutiontemplate->setVariable("FEEDBACK", ilLegacyFormElementsUtil::prepareTextareaOutput($feedback, true));
        }

        $solutiontemplate->setVariable("SOLUTION_OUTPUT", $questionoutput);

        $solutionoutput = $solutiontemplate->get();
        if (!$show_question_only) {
            // get page object output
            $solutionoutput = $this->getILIASPage($solutionoutput);
        }
        return $solutionoutput;
    }

    public function getAutoSavedSolutionOutput(
        int $active_id,
        ?int $pass = null,
        bool $graphical_output = false,
        bool $result_output = false,
        bool $show_question_only = true,
        bool $show_feedback = false,
        bool $show_correct_solution = false,
        bool $show_manual_scoring = false,
        bool $show_question_text = true,
        bool $show_autosave_title = false,
        bool $show_inline_feedback = false,
    ): string {
        $user_solution = $this->getUserAnswer($active_id, $pass);

        if (($active_id > 0) && (!$show_correct_solution)) {
            $solution = $user_solution;
        } else {
            $solution = $this->getBestAnswer($this->renderPurposeSupportsFormHtml());
        }

        $template = new ilTemplate("tpl.il_as_qpl_text_question_output_solution.html", true, true, "components/ILIAS/TestQuestionPool");
        $solutiontemplate = new ilTemplate("tpl.il_as_tst_solution_output.html", true, true, "components/ILIAS/TestQuestionPool");

        $solution = '';
        $autosaved_solution = $this->object->getLatestAutosaveContent($active_id, $pass);
        if ($autosaved_solution !== null) {
            if ($show_autosave_title) {
                $template->setCurrentBlock('autosave_title');
                $template->setVariable('AUTOSAVE_TITLE', $this->lng->txt('autosavecontent'));
                $template->parseCurrentBlock();
            }
            $solution = $this->object->getHtmlUserSolutionPurifier()->purify($autosaved_solution);
            if ($this->renderPurposeSupportsFormHtml()) {
                $template->setCurrentBlock('essay_div');
                $template->setVariable("DIV_ESSAY", ilLegacyFormElementsUtil::prepareTextareaOutput($solution, true));
            } else {
                $template->setCurrentBlock('essay_textarea');
                $template->setVariable("TA_ESSAY", ilLegacyFormElementsUtil::prepareTextareaOutput($solution, true, true));
            }
            $template->parseCurrentBlock();
        }

        if (!$show_correct_solution) {
            $max_no_of_chars = $this->object->getMaxNumOfChars();

            if ($max_no_of_chars == 0) {
                $max_no_of_chars = ucfirst($this->lng->txt('unlimited'));
            }

            $act_no_of_chars = $this->object->countLetters($solution);
            $template->setVariable("CHARACTER_INFO", '<b>' . $max_no_of_chars . '</b>' .
                $this->lng->txt('answer_characters') . ' <b>' . $act_no_of_chars . '</b>');

            if ($this->object->isWordCounterEnabled()) {
                $template->setCurrentBlock('word_count');
                $template->setVariable(
                    'WORD_COUNT',
                    $this->lng->txt('qst_essay_written_words') .
                    ' <b>' . $this->object->countWords($solution) . '</b>'
                );
                $template->parseCurrentBlock();
            }
        }
        if ($show_question_text == true) {
            $template->setVariable("QUESTIONTEXT", $this->object->getQuestionForHTMLOutput());
        }
        $questionoutput = $template->get();

        $feedback = '';
        if ($show_feedback) {
            if (!$this->isTestPresentationContext()) {
                $fb = $this->getGenericFeedbackOutput((int) $active_id, $pass);
                $feedback .= strlen($fb) ? $fb : '';
            }
        }
        if (strlen($feedback)) {
            $cssClass = (
                $this->hasCorrectSolution($active_id, $pass) ?
                ilAssQuestionFeedback::CSS_CLASS_FEEDBACK_CORRECT : ilAssQuestionFeedback::CSS_CLASS_FEEDBACK_WRONG
            );

            $solutiontemplate->setVariable("ILC_FB_CSS_CLASS", $cssClass);
            $solutiontemplate->setVariable("FEEDBACK", ilLegacyFormElementsUtil::prepareTextareaOutput($feedback, true));
        }

        $solutiontemplate->setVariable("SOLUTION_OUTPUT", $questionoutput);

        $solutionoutput = $solutiontemplate->get();
        if (!$show_question_only) {
            // get page object output
            $solutionoutput = $this->getILIASPage($solutionoutput);
        }
        return $solutionoutput;
    }

    private function getBestAnswer($asHtml): string
    {
        $answers = $this->object->getAnswers();
        if (!count($answers)) {
            return '';
        }

        if ($asHtml) {
            $tplFile = 'tpl.il_as_qpl_text_question_best_solution_html.html';
        } else {
            $tplFile = 'tpl.il_as_qpl_text_question_best_solution_ta.html';
        }

        $tpl = new ilTemplate($tplFile, true, true, 'components/ILIAS/TestQuestionPool');

        foreach ($answers as $answer) {
            $keywordString = '';
            if (in_array($this->object->getKeywordRelation(), assTextQuestion::getScoringModesWithPointsByKeyword())) {
                $keywordString .= (string) $answer->getPoints() . ' ';
                if ($answer->getPoints() === 1.0 || $answer->getPoints() === -1.0) {
                    $keywordString .= $this->lng->txt('point');
                } else {
                    $keywordString .= $this->lng->txt('points');
                }
                $keywordString .= ' ' . $this->lng->txt('for') . ' ';
            }
            $keywordString .= $answer->getAnswertext();

            $tpl->setCurrentBlock('keyword');
            $tpl->setVariable('KEYWORD', $keywordString);
            $tpl->parseCurrentBlock();
        }

        $tpl->setVariable('KEYWORD_HEADER', $this->lng->txt('solution_contain_keywords'));
        $tpl->setVariable('SCORING_LABEL', $this->lng->txt('essay_scoring_mode') . ': ');

        switch ($this->object->getKeywordRelation()) {
            case assTextQuestion::SCORING_MODE_KEYWORD_RELATION_ANY:
                $tpl->setVariable('SCORING_MODE', $this->lng->txt('essay_scoring_mode_keyword_relation_any'));
                break;
            case assTextQuestion::SCORING_MODE_KEYWORD_RELATION_ALL:
                $tpl->setVariable('SCORING_MODE', $this->lng->txt('essay_scoring_mode_keyword_relation_all'));
                break;
            case assTextQuestion::SCORING_MODE_KEYWORD_RELATION_ONE:
                $tpl->setVariable('SCORING_MODE', $this->lng->txt('essay_scoring_mode_keyword_relation_one'));
                break;
        }

        return $tpl->get();
    }

    private function getUserAnswer($active_id, $pass)
    {
        $user_solution = "";
        $solutions = $this->object->getSolutionValues($active_id, $pass);
        foreach ($solutions as $idx => $solution_value) {
            $user_solution = $solution_value["value1"];
        }
        return $user_solution;
    }

    public function getPreview(
        bool $show_question_only = false,
        bool $show_inline_feedback = false
    ): string {
        $template = new ilTemplate("tpl.il_as_qpl_text_question_output.html", true, true, "components/ILIAS/TestQuestionPool");
        if ($this->object->getMaxNumOfChars()) {
            $template->setCurrentBlock("maximum_char_hint");
            $template->setVariable("MAXIMUM_CHAR_HINT", sprintf($this->lng->txt("text_maximum_chars_allowed"), $this->object->getMaxNumOfChars()));
            $template->parseCurrentBlock();
            #mbecker: No such block. $template->setCurrentBlock("has_maxchars");
            $template->setVariable("MAXCHARS", $this->object->getMaxNumOfChars());
            $template->parseCurrentBlock();
            $template->setCurrentBlock("maxchars_counter");
            $template->setVariable("QID", $this->object->getId());
            $template->setVariable("MAXCHARS", $this->object->getMaxNumOfChars());
            $template->setVariable("TEXTBOXSIZE", strlen($this->object->getMaxNumOfChars()));
            $template->setVariable("CHARACTERS", $this->lng->txt("qst_essay_chars_remaining"));
            $template->parseCurrentBlock();
        }

        if ($this->object->isWordCounterEnabled()) {
            $template->setCurrentBlock("word_counter");
            $template->setVariable("QID", $this->object->getId());
            $template->setVariable("WORDCOUNTER", $this->lng->txt("qst_essay_allready_written_words"));
            $template->parseCurrentBlock();
        }

        if (is_object($this->getPreviewSession())) {
            $template->setVariable(
                "ESSAY",
                ilLegacyFormElementsUtil::prepareFormOutput(
                    (string) $this->getPreviewSession()->getParticipantsSolution()
                )
            );
        }

        $template->setVariable("QUESTIONTEXT", $this->object->getQuestionForHTMLOutput());
        $template->setVariable("QID", $this->object->getId());

        $questionoutput = $template->get();

        $questionoutput .= $this->getJsCode();

        if (!$show_question_only) {
            // get page object output
            $questionoutput = $this->getILIASPage($questionoutput);
        }
        return $questionoutput;
    }

    public function getTestOutput(
        int $active_id,
        int $pass,
        bool $is_question_postponed = false,
        array|bool $user_post_solutions = false,
        bool $show_specific_inline_feedback = false
    ): string {
        // get the solution of the user for the active pass or from the last pass if allowed
        $user_solution = "";
        if ($active_id) {
            $solutions = $this->object->getUserSolutionPreferingIntermediate($active_id, $pass);
            foreach ($solutions as $solution_value) {
                $user_solution = $solution_value["value1"];
            }

            if ($this->tiny_mce_enabled) {
                $user_solution = htmlentities($user_solution);
            }

            $user_solution = str_replace(['{', '}', '\\'], ['&#123;', '&#125;', '&#92;'], $user_solution);
        }

        $template = new ilTemplate("tpl.il_as_qpl_text_question_output.html", true, true, "components/ILIAS/TestQuestionPool");
        if ($this->object->getMaxNumOfChars()) {
            $template->setCurrentBlock("maximum_char_hint");
            $template->setVariable("MAXIMUM_CHAR_HINT", sprintf($this->lng->txt("text_maximum_chars_allowed"), $this->object->getMaxNumOfChars()));
            $template->parseCurrentBlock();
            $template->setVariable("MAXCHARS", $this->object->getMaxNumOfChars());
            $template->parseCurrentBlock();
            $template->setCurrentBlock("maxchars_counter");
            $template->setVariable("MAXCHARS", $this->object->getMaxNumOfChars());
            $template->setVariable("QID", $this->object->getId());
            $template->setVariable("TEXTBOXSIZE", strlen($this->object->getMaxNumOfChars()));
            $template->setVariable("CHARACTERS", $this->lng->txt("qst_essay_chars_remaining"));
            $template->parseCurrentBlock();
        }

        if ($this->object->isWordCounterEnabled()) {
            $template->setCurrentBlock("word_counter");
            $template->setVariable("QID", $this->object->getId());
            $template->setVariable("WORDCOUNTER", $this->lng->txt("qst_essay_allready_written_words"));
            $template->parseCurrentBlock();
        }

        $template->setVariable("QID", $this->object->getId());
        $template->setVariable("ESSAY", $user_solution);
        $template->setVariable("QUESTIONTEXT", $this->object->getQuestionForHTMLOutput());
        $questionoutput = $template->get();

        $questionoutput .= $this->getJsCode();

        $pageoutput = $this->outQuestionPage("", $is_question_postponed, $active_id, $questionoutput);
        return $pageoutput;
    }

    protected function getJsCode(): string
    {
        $tpl = new ilTemplate('tpl.charcounter.html', true, true, 'components/ILIAS/TestQuestionPool');

        $tpl->setCurrentBlock('tinymce_handler');
        $tpl->touchBlock('tinymce_handler');
        $tpl->parseCurrentBlock();

        if ($this->object->getMaxNumOfChars() > 0) {
            $tpl->setCurrentBlock('letter_counter_js');
            $tpl->setVariable("MAXCHARS", $this->object->getMaxNumOfChars());
            $tpl->parseCurrentBlock();
        }

        if ($this->object->isWordCounterEnabled()) {
            $tpl->setCurrentBlock('word_counter_js');
            $tpl->touchBlock('word_counter_js');
            $tpl->parseCurrentBlock();
        }

        $tpl->setCurrentBlock('counter_js');
        $tpl->setVariable("QID", $this->object->getId());
        $tpl->parseCurrentBlock();

        $this->tpl->addOnLoadCode('mcQuestionCharCounterInit();');
        return $tpl->get();
    }

    public function addSuggestedSolution(): void
    {
        $this->setAdditionalContentEditingModeFromPost();
        ilSession::set('subquestion_index', 0);
        $cmd = $this->request_data_collector->rawArray('cmd');

        if ($cmd['addSuggestedSolution'] && $this->writePostData()) {
            $this->tpl->setOnScreenMessage('info', $this->getErrorMessage());
            $this->editQuestion();
            return;
        }
        $this->object->saveToDb();
        $this->ctrl->setParameter($this, 'q_id', $this->object->getId());
        $this->tpl->setVariable('HEADER', $this->object->getTitleForHTMLOutput());
        $this->getQuestionTemplate();
    }

    public function getSpecificFeedbackOutput(array $userSolution): string
    {
        return '';
    }

    public function writeQuestionSpecificPostData(ilPropertyFormGUI $form): void
    {
        $this->object->setWordCounterEnabled($this->request_data_collector->bool('wordcounter') ?? false);
        $this->object->setMaxNumOfChars($this->request_data_collector->int('maxchars'));
        $this->object->setTextRating($this->request_data_collector->string('text_rating'));
        $this->object->setKeywordRelation($this->request_data_collector->string('scoring_mode'));
    }

    public function writeAnswerSpecificPostData(ilPropertyFormGUI $form): void
    {
        $points = 0;
        switch ($this->object->getKeywordRelation()) {
            case assTextQuestion::SCORING_MODE_KEYWORD_RELATION_NONE:
                $this->object->setAnswers([]);
                $points = $this->request_data_collector->float('non_keyword_points');
                break;
            case assTextQuestion::SCORING_MODE_KEYWORD_RELATION_ANY:
                $this->object->setAnswers($this->request_data_collector->raw('any_keyword'));
                $points = $this->object->getMaximumPoints();
                break;
            case assTextQuestion::SCORING_MODE_KEYWORD_RELATION_ALL:
                $this->object->setAnswers($this->request_data_collector->raw('all_keyword'));
                $points = $this->request_data_collector->float('all_keyword_points');
                break;
            case assTextQuestion::SCORING_MODE_KEYWORD_RELATION_ONE:
                $this->object->setAnswers($this->request_data_collector->raw('one_keyword'));
                $points = $this->request_data_collector->float('one_keyword_points');
                break;
        }
        $this->object->setPoints((float) $points);
    }

    public function populateQuestionSpecificFormPart(\ilPropertyFormGUI $form): ilPropertyFormGUI
    {
        // wordcounter
        $wordcounter = new ilCheckboxInputGUI($this->lng->txt('qst_essay_wordcounter_enabled'), 'wordcounter');
        $wordcounter->setInfo($this->lng->txt('qst_essay_wordcounter_enabled_info'));
        $wordcounter->setChecked($this->object->isWordCounterEnabled());
        $form->addItem($wordcounter);

        // maxchars
        $maxchars = new ilNumberInputGUI($this->lng->txt("maxchars"), "maxchars");
        $maxchars->setSize(5);
        $maxchars->setMinValue(1);
        if ($this->object->getMaxNumOfChars() > 0) {
            $maxchars->setValue($this->object->getMaxNumOfChars());
        }
        $maxchars->setInfo($this->lng->txt("description_maxchars"));
        $form->addItem($maxchars);

        // text rating
        $textrating = new ilSelectInputGUI($this->lng->txt("text_rating"), "text_rating");
        $text_options = [
            "ci" => $this->lng->txt("cloze_textgap_case_insensitive"),
            "cs" => $this->lng->txt("cloze_textgap_case_sensitive"),
            "l1" => sprintf($this->lng->txt("cloze_textgap_levenshtein_of"), "1"),
            "l2" => sprintf($this->lng->txt("cloze_textgap_levenshtein_of"), "2"),
            "l3" => sprintf($this->lng->txt("cloze_textgap_levenshtein_of"), "3"),
            "l4" => sprintf($this->lng->txt("cloze_textgap_levenshtein_of"), "4"),
            "l5" => sprintf($this->lng->txt("cloze_textgap_levenshtein_of"), "5")
        ];
        $textrating->setOptions($text_options);
        $textrating->setValue($this->object->getTextRating());
        $form->addItem($textrating);

        return $form;
    }

    public function populateAnswerSpecificFormPart(\ilPropertyFormGUI $form): ilPropertyFormGUI
    {
        $scoringMode = new ilRadioGroupInputGUI(
            $this->lng->txt('essay_scoring_mode'),
            'scoring_mode'
        );

        $scoringOptionNone = new ilRadioOption(
            $this->lng->txt('essay_scoring_mode_without_keywords'),
            assTextQuestion::SCORING_MODE_KEYWORD_RELATION_NONE,
            $this->lng->txt(
                'essay_scoring_mode_without_keywords_desc'
            )
        );
        $scoringOptionAnyKeyword = new ilRadioOption(
            $this->lng->txt('essay_scoring_mode_keyword_relation_any'),
            assTextQuestion::SCORING_MODE_KEYWORD_RELATION_ANY,
            $this->lng->txt(
                'essay_scoring_mode_keyword_relation_any_desc'
            )
        );
        $scoringOptionAllKeyword = new ilRadioOption(
            $this->lng->txt('essay_scoring_mode_keyword_relation_all'),
            assTextQuestion::SCORING_MODE_KEYWORD_RELATION_ALL,
            $this->lng->txt(
                'essay_scoring_mode_keyword_relation_all_desc'
            )
        );
        $scoringOptionOneKeyword = new ilRadioOption(
            $this->lng->txt('essay_scoring_mode_keyword_relation_one'),
            assTextQuestion::SCORING_MODE_KEYWORD_RELATION_ONE,
            $this->lng->txt(
                'essay_scoring_mode_keyword_relation_one_desc'
            )
        );

        $scoringMode->addOption($scoringOptionNone);
        $scoringMode->addOption($scoringOptionAnyKeyword);
        $scoringMode->addOption($scoringOptionAllKeyword);
        $scoringMode->addOption($scoringOptionOneKeyword);
        $scoringMode->setRequired(true);
        $scoringMode->setValue($this->object->getKeywordRelation());

        if ($this->object->getAnswerCount() == 0) {
            $this->object->addAnswer("", 1, 0, 0);
        }

        // Without Keywords
        $nonKeywordPoints = new ilNumberInputGUI($this->lng->txt("points"), "non_keyword_points");
        $nonKeywordPoints->allowDecimals(true);
        $nonKeywordPoints->setValue($this->object->getPoints());
        $nonKeywordPoints->setRequired(true);
        $nonKeywordPoints->setSize(3);
        $nonKeywordPoints->setMinValue(0.0);
        $nonKeywordPoints->setMinvalueShouldBeGreater(true);
        $scoringOptionNone->addSubItem($nonKeywordPoints);

        // Any Keyword
        $anyKeyword = new ilEssayKeywordWizardInputGUI($this->lng->txt("answers"), "any_keyword");
        $anyKeyword->setRequired(true);
        $anyKeyword->setQuestionObject($this->object);
        $anyKeyword->setSingleline(true);
        $anyKeyword->setValues($this->object->getAnswers());
        $scoringOptionAnyKeyword->addSubItem($anyKeyword);

        // All Keywords
        $allKeyword = new ilTextWizardInputGUI($this->lng->txt("answers"), "all_keyword");
        $allKeyword->setRequired(true);
        //$allKeyword->setQuestionObject($this->object);
        //$allKeyword->setSingleline(TRUE);
        $allKeyword->setValues(self::buildAnswerTextOnlyArray($this->object->getAnswers()));
        $allKeyword->setMaxLength($anyKeyword->getMaxLength());
        $scoringOptionAllKeyword->addSubItem($allKeyword);
        $allKeywordPoints = new ilNumberInputGUI($this->lng->txt("points"), "all_keyword_points");
        $allKeywordPoints->allowDecimals(true);
        $allKeywordPoints->setValue($this->object->getPoints());
        $allKeywordPoints->setRequired(true);
        $allKeywordPoints->setSize(3);
        $allKeywordPoints->setMinValue(0.0);
        $allKeywordPoints->setMinvalueShouldBeGreater(true);
        $scoringOptionAllKeyword->addSubItem($allKeywordPoints);

        // One Keywords
        $oneKeyword = new ilTextWizardInputGUI($this->lng->txt("answers"), "one_keyword");
        $oneKeyword->setRequired(true);
        //$oneKeyword->setQuestionObject($this->object);
        //$oneKeyword->setSingleline(TRUE);
        $oneKeyword->setValues(self::buildAnswerTextOnlyArray($this->object->getAnswers()));
        $oneKeyword->setMaxLength($anyKeyword->getMaxLength());
        $scoringOptionOneKeyword->addSubItem($oneKeyword);
        $oneKeywordPoints = new ilNumberInputGUI($this->lng->txt("points"), "one_keyword_points");
        $oneKeywordPoints->allowDecimals(true);
        $oneKeywordPoints->setValue($this->object->getPoints());
        $oneKeywordPoints->setRequired(true);
        $oneKeywordPoints->setSize(3);
        $oneKeywordPoints->setMinValue(0.0);
        $oneKeywordPoints->setMinvalueShouldBeGreater(true);
        $scoringOptionOneKeyword->addSubItem($oneKeywordPoints);

        $form->addItem($scoringMode);
        return $form;
    }

    /**
     * Returns a list of postvars which will be suppressed in the form output when used in scoring adjustment.
     * The form elements will be shown disabled, so the users see the usual form but can only edit the settings, which
     * make sense in the given context.
     *
     * E.g. array('cloze_type', 'image_filename')
     *
     * @return string[]
     */
    public function getAfterParticipationSuppressionAnswerPostVars(): array
    {
        return [];
    }

    /**
     * Returns a list of postvars which will be suppressed in the form output when used in scoring adjustment.
     * The form elements will be shown disabled, so the users see the usual form but can only edit the settings, which
     * make sense in the given context.
     *
     * E.g. array('cloze_type', 'image_filename')
     *
     * @return string[]
     */
    public function getAfterParticipationSuppressionQuestionPostVars(): array
    {
        return [];
    }

    public function isAnswerFreuqencyStatisticSupported(): bool
    {
        return false;
    }

    public function getAnswersFrequency($relevantAnswers, $questionIndex): array
    {
        return [];
    }

    public function populateCorrectionsFormProperties(ilPropertyFormGUI $form): void
    {
        $this->populateQuestionSpecificFormPart($form);
        $this->populateAnswerSpecificFormPart($form);

        $form->removeItemByPostVar('maxchars');
    }

    /**
     * @param ilPropertyFormGUI $form
     */
    public function saveCorrectionsFormProperties(ilPropertyFormGUI $form): void
    {
        $this->writeQuestionSpecificPostData($form);
        $this->writeAnswerSpecificPostData($form);
    }
}
