<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

namespace ILIAS\News\Data;

use DateTimeImmutable;

/**
 * News Criteria DTO for querying news items supports caching, JSON serialization, and validation
 */
final class NewsCriteria implements \JsonSerializable
{
    public function __construct(
        private ?DateTimeImmutable $start_date = null,
        private ?int $period = null,
        private bool $only_public = false,
        private ?int $min_priority = null,
        private ?int $max_priority = null,
        private ?int $limit = null,
        private bool $prevent_nesting = false,
        private bool $no_auto_generated = false,
        private array $excluded_news_ids = [],
        private ?bool $include_read_status = null,
        private ?int $read_user_id = null,
    ) {
        // By default, include read status if only public is not set
        $this->include_read_status = $this->include_read_status ?? !$this->only_public;
    }

    /*
        Getters and Setters
     */

    public function getStartDate(): ?DateTimeImmutable
    {
        return $this->start_date;
    }

    public function getPeriod(): ?int
    {
        return $this->period;
    }

    public function isOnlyPublic(): bool
    {
        return $this->only_public;
    }

    public function getMinPriority(): ?int
    {
        return $this->min_priority;
    }

    public function getMaxPriority(): ?int
    {
        return $this->max_priority;
    }

    public function getLimit(): ?int
    {
        return $this->limit;
    }

    public function isPreventNesting(): bool
    {
        return $this->prevent_nesting;
    }

    public function isNoAutoGenerated(): bool
    {
        return $this->no_auto_generated;
    }

    public function getExcludedNewsIds(): array
    {
        return $this->excluded_news_ids;
    }

    public function isIncludeReadStatus(): bool
    {
        return $this->include_read_status ?? false;
    }

    public function getReadUserId(): ?int
    {
        return $this->read_user_id;
    }

    public function withStartDate(?DateTimeImmutable $start_date): self
    {
        $new = clone $this;
        $new->start_date = $start_date;
        return $new;
    }

    public function withPeriod(?int $period): self
    {
        $new = clone $this;
        $new->period = $period;
        return $new;
    }

    public function withOnlyPublic(bool $only_public): self
    {
        $new = clone $this;
        $new->only_public = $only_public;
        return $new;
    }

    public function withMinPriority(?int $min_priority): self
    {
        $new = clone $this;
        $new->min_priority = $min_priority;
        return $new;
    }

    public function withMaxPriority(?int $max_priority): self
    {
        $new = clone $this;
        $new->max_priority = $max_priority;
        return $new;
    }

    public function withLimit(?int $limit): self
    {
        $new = clone $this;
        $new->limit = $limit;
        return $new;
    }

    public function withPreventNesting(bool $stop_nesting): self
    {
        $new = clone $this;
        $new->prevent_nesting = $stop_nesting;
        return $new;
    }

    public function withNoAutoGenerated(bool $no_auto_generated): self
    {
        $new = clone $this;
        $new->no_auto_generated = $no_auto_generated;
        return $new;
    }

    public function withExcludedNewsIds(array $excluded_news_ids): self
    {
        $new = clone $this;
        $new->excluded_news_ids = array_map('intval', $excluded_news_ids);
        return $new;
    }

    public function withIncludeReadStatus(bool $include_read_status): self
    {
        $new = clone $this;
        $new->include_read_status = $include_read_status;
        return $new;
    }

    public function withReadUserId(?int $read_user_id): self
    {
        $new = clone $this;
        $new->read_user_id = $read_user_id;
        return $new;
    }

    /*
        Public Methods
     */

    public function jsonSerialize(): array
    {
        return $this->toArray();
    }

    public function toArray(): array
    {
        return [
            'start_date' => $this->start_date?->format('Y-m-d H:i:s'),
            'period' => $this->period,
            'only_public' => $this->only_public,
            'min_priority' => $this->min_priority,
            'max_priority' => $this->max_priority,
            'limit' => $this->limit,
            'stop_nesting' => $this->prevent_nesting,
            'no_auto_generated' => $this->no_auto_generated,
            'excluded_news_ids' => $this->excluded_news_ids,
            'include_read_status' => $this->include_read_status,
            'read_user_id' => $this->read_user_id,
        ];
    }

    /**
     * Check if criteria has priority filters
     */
    public function hasPriorityFilters(): bool
    {
        return $this->min_priority !== null || $this->max_priority !== null;
    }

    /**
     * Validate criteria parameters
     */
    public function validate(): void
    {
        if ($this->min_priority !== null && $this->max_priority !== null && $this->min_priority > $this->max_priority) {
            throw new \InvalidArgumentException('Min priority cannot be greater than max priority');
        }

        if ($this->limit !== null && $this->limit < 0) {
            throw new \InvalidArgumentException('Limit cannot be negative');
        }
    }
}
