<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

use ILIAS\UI\Factory;
use ILIAS\UI\Renderer;
use ILIAS\UI\Component\Dropdown\Standard;
use ILIAS\UI\Component\Item\Item;
use ILIAS\UI\Component\Modal\RoundTrip;
use ILIAS\Data\Factory as DataFactory;
use ILIAS\Forum\Drafts\ForumDraftsTable;
use ILIAS\Forum\Notification\NotificationType;

/**
 * @ilCtrl_Calls ilObjForumGUI: ilPermissionGUI, ilForumExportGUI, ilInfoScreenGUI
 * @ilCtrl_Calls ilObjForumGUI: ilColumnGUI, ilPublicUserProfileGUI, ilForumModeratorsGUI, ilRepositoryObjectSearchGUI
 * @ilCtrl_Calls ilObjForumGUI: ilObjectCopyGUI, ilExportGUI, ilCommonActionDispatcherGUI, ilRatingGUI
 * @ilCtrl_Calls ilObjForumGUI: ilForumSettingsGUI, ilContainerNewsSettingsGUI, ilLearningProgressGUI, ilForumPageGUI
 * @ilCtrl_Calls ilObjForumGUI: ilObjectContentStyleSettingsGUI
 */
class ilObjForumGUI extends ilObjectGUI implements ilDesktopItemHandling, ilForumObjectConstants, ilCtrlSecurityInterface
{
    use ilForumRequestTrait;

    private array $viewModeOptions = [
        ilForumProperties::VIEW_TREE => 'sort_by_posts',
        ilForumProperties::VIEW_DATE_ASC => 'sort_by_date',
    ];

    private array $sortationOptions = [
        ilForumProperties::VIEW_DATE_ASC => 'ascending_order',
        ilForumProperties::VIEW_DATE_DESC => 'descending_order',
    ];

    private \ILIAS\GlobalScreen\Services $globalScreen;
    public string $modal_history = '';
    public ilForumProperties $objProperties;
    private ilForumTopic $objCurrentTopic;
    private ilForumPost $objCurrentPost;
    private bool $display_confirm_post_activation = false;
    private bool $is_moderator;
    private ?ilPropertyFormGUI $replyEditForm = null;
    private bool $hideToolbar = false;
    private \Psr\Http\Message\ServerRequestInterface $httpRequest;
    private \ILIAS\HTTP\Services $http;
    private Factory $uiFactory;
    private Renderer $uiRenderer;
    private ?array $forumObjects = null;
    private string $confirmation_gui_html = '';
    private ilForumSettingsGUI $forum_settings_gui;
    public ilNavigationHistory $ilNavigationHistory;
    private string $requestAction;
    private array $modalActionsContainer = [];

    public ilObjectDataCache $ilObjDataCache;
    public \ILIAS\DI\RBACServices $rbac;
    public ilHelpGUI $ilHelp;
    private Factory $factory;
    private Renderer $renderer;

    private int $selectedSorting;
    private ilForumThreadSettingsSessionStorage $selected_post_storage;
    protected \ILIAS\Style\Content\Object\ObjectFacade $content_style_domain;
    protected \ILIAS\Style\Content\GUIService $content_style_gui;
    private array $modal_collection = [];
    private readonly bool $in_page_editor_style_context;

    public function __construct($data, int $id = 0, bool $call_by_reference = true, bool $prepare_output = true)
    {
        global $DIC;

        $this->httpRequest = $DIC->http()->request();
        $this->http = $DIC->http();

        $this->uiFactory = $DIC->ui()->factory();
        $this->uiRenderer = $DIC->ui()->renderer();
        $this->globalScreen = $DIC->globalScreen();

        $this->ilObjDataCache = $DIC['ilObjDataCache'];
        $this->ilNavigationHistory = $DIC['ilNavigationHistory'];
        $this->ilHelp = $DIC['ilHelp'];
        $this->rbac = $DIC->rbac();
        $this->factory = $DIC->ui()->factory();
        $this->renderer = $DIC->ui()->renderer();

        $this->type = 'frm';
        parent::__construct($data, $id, $call_by_reference, false);
        $this->ctrl->saveParameter($this, ['ref_id']);
        $this->tpl->addJavaScript('assets/js/Basic.js');

        $this->lng->loadLanguageModule('forum');
        $this->lng->loadLanguageModule('content');

        $ref_id = $this->retrieveIntOrZeroFrom($this->http->wrapper()->query(), 'ref_id');

        $this->objProperties = ilForumProperties::getInstance($this->ilObjDataCache->lookupObjId($ref_id));
        $this->is_moderator = $this->access->checkAccess('moderate_frm', '', $ref_id);

        $this->objCurrentTopic = new ilForumTopic($this->retrieveThrPk(), $this->is_moderator);
        $this->requestAction = (string) ($this->httpRequest->getQueryParams()['action'] ?? '');
        $cs = $DIC->contentStyle();
        $this->content_style_gui = $cs->gui();
        if (is_object($this->object)) {
            $this->content_style_domain = $cs->domain()->styleForRefId($this->object->getRefId());
        }
        $this->in_page_editor_style_context = $this->http->wrapper()->query()->has('page_editor_style');
        $this->ctrl->saveParameterByClass(ilObjectContentStyleSettingsGUI::class, 'page_editor_style');

        $this->initSessionStorage();
    }

    protected function initSessionStorage(): void
    {
        $forumValues = ilSession::get('frm');
        if (!is_array($forumValues)) {
            $forumValues = [];
            ilSession::set('frm', $forumValues);
        }

        $threadId = $this->objCurrentTopic->getId();
        if ($threadId > 0 && !isset($forumValues[$threadId])) {
            $forumValues[(int) $threadId] = [];
            ilSession::set('frm', $forumValues);
        }

        $this->selected_post_storage = new ilForumThreadSettingsSessionStorage('frm_selected_post');
    }

    private function retrieveThrPk(): int
    {
        return $this->retrieveIntOrZeroFrom($this->http->wrapper()->query(), 'thr_pk');
    }

    private function retrieveDraftId(bool $from_post = false): int
    {
        return $this->retrieveIntOrZeroFrom(
            $from_post ? $this->http->wrapper()->post() : $this->http->wrapper()->query(),
            'draft_id'
        );
    }

    protected function toggleExplorerNodeStateObject(): void
    {
        $exp = new ilForumExplorerGUI(
            'frm_exp_' . $this->objCurrentTopic->getId(),
            $this,
            'viewThread',
            $this->objCurrentTopic,
            $this->objCurrentTopic->getPostRootNode($this->is_moderator)
        );
        $exp->toggleExplorerNodeState();
    }

    protected function ensureValidPageForCurrentPosting(
        array $subtree_nodes,
        array $pagedPostings,
        int $pageSize,
        ilForumPost $firstForumPost
    ): void {
        if ($firstForumPost->getId() === $this->objCurrentPost->getId()) {
            return;
        }

        if ($subtree_nodes !== [] && $this->objCurrentPost->getId() > 0) {
            $isCurrentPostingInPage = array_filter($pagedPostings, fn(ilForumPost $posting): bool => (
                $posting->getId() === $this->objCurrentPost->getId()
            ));

            if ([] === $isCurrentPostingInPage) {
                $pageOfCurrentPosting = 0;
                $i = 0;
                foreach ($subtree_nodes as $node) {
                    if ($i > 0 && $i % $pageSize === 0) {
                        ++$pageOfCurrentPosting;
                    }

                    if ($node->getId() === $this->objCurrentPost->getId()) {
                        break;
                    }

                    ++$i;
                }

                $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
                $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
                $this->ctrl->setParameter($this, 'page', $pageOfCurrentPosting);
                $this->ctrl->setParameter(
                    $this,
                    'orderby',
                    $this->getOrderByParam()
                );
                $this->ctrl->redirect($this, 'viewThread', (string) $this->objCurrentPost->getId());
            }
        }
    }

    public function ensureThreadBelongsToForum(int $objId, ilForumTopic $thread): void
    {
        $forumId = ilObjForum::lookupForumIdByObjId($objId);
        if ($thread->getForumId() !== $forumId) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }
    }

    private function decorateWithAutosave(ilPropertyFormGUI $form): void
    {
        if (ilForumPostDraft::isAutoSavePostDraftAllowed()) {
            $interval = ilForumPostDraft::lookupAutosaveInterval();

            $this->tpl->addJavaScript('assets/js/autosave_forum.js');
            $autosave_cmd = 'autosaveDraftAsync';
            if ($this->objCurrentPost->getId() === 0 && $this->objCurrentPost->getThreadId() === 0) {
                $autosave_cmd = 'autosaveThreadDraftAsync';
            }
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
            $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
            $this->ctrl->setParameter($this, 'draft_id', $this->retrieveDraftId());
            $this->ctrl->setParameter($this, 'action', ilUtil::stripSlashes($this->requestAction));
            $this->tpl->addOnLoadCode(
                "il.Language.setLangVar('saving', " . json_encode($this->lng->txt('saving'), JSON_THROW_ON_ERROR) . ');'
            );

            $this->tpl->addOnLoadCode('il.ForumDraftsAutosave.init(' . json_encode([
                'loadingImgSrc' => ilUtil::getImagePath('media/loader.svg'),
                'draftId' => $this->retrieveDraftId(),
                'interval' => $interval * 1000,
                'url' => $this->ctrl->getFormAction($this, $autosave_cmd, '', true),
                'selectors' => [
                    'form' => '#form_' . $form->getId()
                ]
            ], JSON_THROW_ON_ERROR) . ');');
        }
    }

    private function isTopLevelReplyCommand(): bool
    {
        return in_array(
            strtolower($this->ctrl->getCmd() ?? ''),
            array_map('strtolower', ['createTopLevelPost', 'saveTopLevelPost', 'saveTopLevelDraft']),
            true
        );
    }

    public function getUnsafeGetCommands(): array
    {
        return [
            'enableForumNotification',
            'disableForumNotification',
            'toggleThreadNotification',
            'confirmDeleteThreadDrafts'
        ];
    }

    public function getSafePostCommands(): array
    {
        return [];
    }

    public function executeCommand(): void
    {
        $next_class = $this->ctrl->getNextClass($this) ?? '';
        $cmd = $this->ctrl->getCmd() ?? '';

        $exclude_cmds = [
            'viewThread',
            'markPostUnread',
            'markPostRead',
            'showThreadNotification',
            'performPostActivation',
            'askForPostActivation',
            'askForPostDeactivation',
            'toggleThreadNotification',
            'toggleStickiness',
            'cancelPost',
            'savePost',
            'saveTopLevelPost',
            'createTopLevelPost',
            'saveTopLevelDraft',
            'quotePost',
            'autosaveDraftAsync',
            'autosaveThreadDraftAsync',
            'saveAsDraft',
            'editDraft',
            'updateDraft',
            'deliverDraftZipFile',
            'deliverZipFile',
            'cancelDraft',
            'deleteThreadDrafts',
            'deletePosting',
            'deletePostingDraft',
            'revokeCensorship',
            'addCensorship',
            'confirmDeleteThreadsObject'
        ];

        if (!in_array($cmd, $exclude_cmds, true)) {
            $this->prepareOutput();
        }

        if (!$this->creation_mode) {
            $this->checkUsersViewMode();
            $pos_pk = $this->retrieveIntOrZeroFrom($this->http->wrapper()->query(), 'pos_pk');
            if ($this->selectedSorting === ilForumProperties::VIEW_TREE &&
                $this->selected_post_storage->get($this->objCurrentTopic->getId()) > 0) {
                $this->objCurrentPost = new ilForumPost(
                    $this->selected_post_storage->get($this->objCurrentTopic->getId()) ?? 0,
                    $this->is_moderator
                );
                $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
            } else {
                $this->selected_post_storage->set($this->objCurrentTopic->getId(), 0);
                $this->objCurrentPost = new ilForumPost(
                    $pos_pk,
                    $this->is_moderator
                );
            }
        }

        if (!$this->getCreationMode() && !$this->ctrl->isAsynch() && $this->access->checkAccess(
            'read',
            '',
            $this->object->getRefId()
        )) {
            $this->ilNavigationHistory->addItem(
                $this->object->getRefId(),
                ilLink::_getLink($this->object->getRefId(), 'frm'),
                'frm'
            );
        }

        switch (strtolower($next_class)) {
            case strtolower(ilForumPageGUI::class):
                if (in_array(strtolower($cmd), array_map('strtolower', [
                    self::UI_CMD_COPAGE_DOWNLOAD_FILE,
                    self::UI_CMD_COPAGE_DISPLAY_FULLSCREEN,
                    self::UI_CMD_COPAGE_DOWNLOAD_PARAGRAPH,
                ]), true)
                ) {
                    if (!$this->checkPermissionBool('read')) {
                        $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
                    }
                } elseif (!$this->checkPermissionBool('write') || $this->user->isAnonymous()) {
                    $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
                }

                $this->content_style_gui->addCss($this->tpl, $this->ref_id);
                $this->tpl->setCurrentBlock('SyntaxStyle');
                $this->tpl->setVariable('LOCATION_SYNTAX_STYLESHEET', ilObjStyleSheet::getSyntaxStylePath());
                $this->tpl->parseCurrentBlock();

                /** @var ilObjForum $obj */
                $obj = $this->object;

                $forwarder = new ilForumPageCommandForwarder(
                    $this->http,
                    $this->ctrl,
                    $this->tabs_gui,
                    $this->lng,
                    $obj,
                    $this->user,
                    $this->content_style_domain
                );

                $pageContent = $forwarder->forward();
                if ($pageContent !== '') {
                    $this->tpl->setContent($pageContent);
                }
                break;

            case strtolower(ilLearningProgressGUI::class):
                if (!ilLearningProgressAccess::checkAccess($this->object->getRefId())) {
                    $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
                }

                $this->tabs_gui->activateTab('learning_progress');

                $usrId = $this->user->getId();
                if (
                    isset($this->request->getQueryParams()['user_id']) &&
                    is_numeric($this->request->getQueryParams()['user_id'])
                ) {
                    $usrId = (int) $this->request->getQueryParams()['user_id'];
                }

                $this->ctrl->forwardCommand(new ilLearningProgressGUI(
                    ilLearningProgressBaseGUI::LP_CONTEXT_REPOSITORY,
                    $this->object->getRefId(),
                    $usrId
                ));
                break;

            case strtolower(ilObjectContentStyleSettingsGUI::class):
                $this->checkPermission('write');

                if ($this->in_page_editor_style_context) {
                    $this->tabs_gui->setBackTarget(
                        $this->lng->txt('back'),
                        $this->ctrl->getLinkTarget(new ilForumPageGUI($this->object->getId()), 'edit')
                    );
                } else {
                    $forum_settings_gui = new ilForumSettingsGUI($this, $this->object);
                    $forum_settings_gui->settingsTabs();
                }

                $settings_gui = $this->content_style_gui
                    ->objectSettingsGUIForRefId(
                        null,
                        $this->ref_id
                    );
                $this->ctrl->forwardCommand($settings_gui);
                break;

            case strtolower(ilForumSettingsGUI::class):
                $forum_settings_gui = new ilForumSettingsGUI($this, $this->object);
                $this->ctrl->forwardCommand($forum_settings_gui);
                break;

            case strtolower(ilRepositoryObjectSearchGUI::class):
                $this->addHeaderAction();
                $this->setSideBlocks();
                $this->tabs_gui->activateTab('forums_threads');
                $this->ctrl->setReturn($this, 'view');
                $search_gui = new ilRepositoryObjectSearchGUI(
                    $this->object->getRefId(),
                    $this,
                    'view'
                );
                $this->ctrl->forwardCommand($search_gui);
                break;

            case strtolower(ilPermissionGUI::class):
                $perm_gui = new ilPermissionGUI($this);
                $this->ctrl->forwardCommand($perm_gui);
                break;

            case strtolower(ilForumExportGUI::class):
                $fex_gui = new ilForumExportGUI();
                $this->ctrl->forwardCommand($fex_gui);
                $this->http->close();
                break;

            case strtolower(ilForumModeratorsGUI::class):
                $fm_gui = new ilForumModeratorsGUI($this->object);
                $this->ctrl->forwardCommand($fm_gui);
                break;

            case strtolower(ilInfoScreenGUI::class):
                $this->infoScreen();
                break;

            case strtolower(ilColumnGUI::class):
                $this->showThreadsObject();
                break;

            case strtolower(ilPublicUserProfileGUI::class):
                $user = $this->retrieveIntOrZeroFrom($this->http->wrapper()->query(), 'user');
                $profile_gui = new ilPublicUserProfileGUI($user);
                $add = $this->getUserProfileAdditional($this->object->getRefId(), $user);
                $profile_gui->setAdditional($add);
                $ret = $this->ctrl->forwardCommand($profile_gui);
                $this->tpl->setContent($ret);
                break;

            case strtolower(ilObjectCopyGUI::class):
                $cp = new ilObjectCopyGUI($this);
                $cp->setType('frm');
                $this->ctrl->forwardCommand($cp);
                break;

            case strtolower(ilExportGUI::class):
                $this->tabs_gui->activateTab('export');
                $exp = new ilExportGUI($this);
                $this->ctrl->forwardCommand($exp);
                break;

            case strtolower(ilRatingGUI::class):
                if (!$this->objProperties->isIsThreadRatingEnabled() || $this->user->isAnonymous()) {
                    $this->error->raiseError($this->lng->txt('msg_no_perm_read'), $this->error->MESSAGE);
                }

                if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
                    $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
                }

                $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentTopic);

                $rating_gui = new ilRatingGUI();
                $rating_gui->setObject(
                    $this->object->getId(),
                    $this->object->getType(),
                    $this->objCurrentTopic->getId(),
                    'thread'
                );

                $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
                $this->ctrl->forwardCommand($rating_gui);

                $avg = ilRating::getOverallRatingForObject(
                    $this->object->getId(),
                    $this->object->getType(),
                    $this->objCurrentTopic->getId(),
                    'thread'
                );
                $this->objCurrentTopic->setAverageRating($avg['avg']);
                $this->objCurrentTopic->update();

                $this->ctrl->redirect($this, 'showThreads');

                // no break
            case strtolower(ilCommonActionDispatcherGUI::class):
                $gui = ilCommonActionDispatcherGUI::getInstanceFromAjaxCall();
                $this->ctrl->forwardCommand($gui);
                break;

            case strtolower(ilContainerNewsSettingsGUI::class):
                $this->checkPermission('write');

                $forum_settings_gui = new ilForumSettingsGUI($this, $this->object);
                $forum_settings_gui->settingsTabs();

                $this->lng->loadLanguageModule('cont');
                $news_set_gui = new ilContainerNewsSettingsGUI($this);
                $news_set_gui->setNewsBlockForced(true);
                $news_set_gui->setPublicNotification(true);
                $this->ctrl->forwardCommand($news_set_gui);
                break;

            default:
                if (in_array($cmd, ['close', 'reopen', 'make_topics_non_sticky', 'makesticky', 'editThread', 'move'])) {
                    $cmd = 'performThreadsAction';
                }

                if ($cmd === null || $cmd === '') {
                    $cmd = 'showThreads';
                }

                $cmd .= 'Object';
                $this->$cmd();

                break;
        }

        if (!$this->in_page_editor_style_context &&
            $cmd !== 'viewThreadObject' && $cmd !== 'showUserObject' && !in_array(
                strtolower($next_class),
                array_map('strtolower', [ilForumPageGUI::class]),
                true
            )) {
            $this->addHeaderAction();
        }
    }

    public function infoScreenObject(): void
    {
        $this->infoScreen();
    }

    protected function initEditCustomForm(ilPropertyFormGUI $a_form): void
    {
        $this->forum_settings_gui = new ilForumSettingsGUI($this, $this->object);
        $this->forum_settings_gui->getCustomForm($a_form);
    }

    protected function getEditFormCustomValues(array &$a_values): void
    {
        $this->forum_settings_gui->getCustomValues($a_values);
    }

    protected function updateCustom(ilPropertyFormGUI $form): void
    {
        $this->forum_settings_gui->updateCustomValues($form);
    }

    public function updateThreadObject(): void
    {
        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if ($this->objCurrentTopic->getId() === 0) {
            $this->showThreadsObject();
            return;
        }

        $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentTopic);

        $title = $this->getEditTitleModal($this->objCurrentTopic)->withRequest($this->http->request())->getData();
        if (array_key_exists(0, $title)) {
            $new_subject = $title[0];
            if ($new_subject !== '') {
                $this->objCurrentTopic->setSubject($new_subject);
                $this->objCurrentTopic->updateThreadTitle();
                $this->tpl->setOnScreenMessage('success', $this->lng->txt('saved_successfully'));
            }

            $this->showThreadsObject();
        }
    }

    public function markAllReadObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $this->object->markAllThreadsRead($this->user->getId());
        $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_all_threads_marked_read'));
        $this->showThreadsObject();
    }

    public function showThreadsObject(): void
    {
        $this->setSideBlocks();
        $this->getCenterColumnHTML();
    }

    public function getContent(): string
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $cmd = $this->ctrl->getCmd();
        $frm = $this->object->Forum;
        $frm->setForumId($this->object->getId());
        $frm->setForumRefId($this->object->getRefId());
        $frm->setMDB2Wherecondition('top_frm_fk = %s ', ['integer'], [$frm->getForumId()]);
        // Import information: Topic (variable $topicData) means frm object, not thread
        $frm_object = $frm->getOneTopic();
        if ($frm_object->getTopPk() > 0) {
            $frm->setDbTable('frm_data');
            $frm->setMDB2WhereCondition('top_pk = %s ', ['integer'], [$frm_object->getTopPk()]);
            $frm->updateVisits($frm_object->getTopPk());

            ilChangeEvent::_recordReadEvent(
                $this->object->getType(),
                $this->object->getRefId(),
                $this->object->getId(),
                $this->user->getId()
            );

            if ($cmd !== 'showThreads') {
                $cmd = 'showThreads';
            }
        }
        $this->tpl->setPermanentLink($this->object->getType(), $this->object->getRefId());

        if (!$this->hideToolbar() && $this->access->checkAccess('add_thread', '', $this->object->getRefId())) {
            $btn = $this->uiFactory->button()
                                   ->standard(
                                       $this->lng->txt('forums_new_thread'),
                                       $this->ctrl->getLinkTarget($this, 'createThread')
                                   );
            $this->toolbar->addStickyItem($btn);
        }

        if (ilForumPostDraft::isSavePostDraftAllowed()) {
            $this->addDraftButtonIfDraftsExists($cmd);
        }

        if ($this->confirmation_gui_html === '' && !$this->user->isAnonymous()) {
            $this->toolbar->addButton(
                $this->lng->txt('forums_mark_read'),
                $this->ctrl->getLinkTarget($this, 'markAllRead')
            );
            $this->ctrl->clearParameters($this);
        }

        if (!$this->user->isAnonymous() && $this->access->checkAccess('write', '', $this->ref_id)) {
            $this->lng->loadLanguageModule('cntr');
            $this->toolbar->addComponent(
                $this->uiFactory->button()->standard(
                    $this->lng->txt('cntr_text_media_editor'),
                    $this->ctrl->getLinkTargetByClass(ilForumPageGUI::class, 'edit')
                )
            );
        }

        $tbl = new ilForumThreadObjectTableGUI(
            $this,
            $cmd,
            $this->object->getRefId(),
            $frm_object,
            $this->is_moderator
        );

        $this->initializeThreadSortation($tbl);
        $this->initializeThreadOffsetAndLimit($tbl);

        $this->renderThreadOverview($tbl, $frm, $frm_object);
        return '';
    }

    private function renderThreadOverview(ilForumThreadObjectTableGUI $tbl, ilForum $frm, ForumDto $frm_object): void
    {
        $data_objects = $tbl->setMapper($frm)->fetchDataAnReturnObject();

        $sticky_threads = [];
        $regular_threads = [];

        if (count($data_objects) > 0) {
            foreach ($data_objects as $thread) {
                if (array_key_exists('thread', $thread)) {
                    /** @var ilForumTopic $current_thread */
                    $current_thread = $thread['thread'];
                    $subject = $current_thread->getSubject();
                    if ($current_thread->isClosed()) {
                        $subject .= ' (' . $this->lng->txt('forums_closed') . ')';
                    }

                    $link = $this->getLinkActionForThread(
                        $this->object->getRefId(),
                        $subject,
                        'viewThread',
                        $current_thread->getId()
                    );
                    $actions = $this->getActionsForThreadOverview($this->object->getRefId(), $current_thread);

                    $list_item = $this->factory
                        ->item()
                        ->standard($link)
                        ->withActions($actions)
                        ->withProperties($this->getThreadProperties($current_thread));
                    $list_item = $this->markTopThreadInOverview($current_thread, $list_item);
                    if ($current_thread->isSticky()) {
                        $sticky_threads[] = $list_item;
                    } else {
                        $regular_threads[] = $list_item;
                    }
                }
            }
        }

        $sticky_threads_item_group = null;
        if (count($sticky_threads) > 0) {
            $sticky_threads_item_group = $this->factory->item()->group(
                count($regular_threads) > 0 ? $this->lng->txt('top_thema') : '',
                $sticky_threads
            );
        }

        $regular_threads_item_group = null;
        if (count($regular_threads) > 0) {
            $regular_threads_item_group = $this->factory->item()->group(
                count($sticky_threads) > 0 ? $this->lng->txt('thema') : '',
                $regular_threads
            );
        }

        $url = $this->http->request()->getRequestTarget();
        $current_page = 0;
        if ($this->http->wrapper()->query()->has(ilForumProperties::PAGE_NAME_THREAD_OVERVIEW)) {
            $current_page = $this->http->wrapper()->query()->retrieve(
                ilForumProperties::PAGE_NAME_THREAD_OVERVIEW,
                $this->refinery->kindlyTo()->int()
            );
        }

        $view_controls[] = $this->getSortationViewControl();
        $view_controls[] = $this->factory
            ->viewControl()
            ->pagination()
            ->withTargetURL($url, ilForumProperties::PAGE_NAME_THREAD_OVERVIEW)
            ->withTotalEntries($frm_object->getTopNumThreads())
            ->withPageSize(ilForumProperties::PAGE_SIZE_THREAD_OVERVIEW)
            ->withMaxPaginationButtons(5)
            ->withCurrentPage($current_page);

        $item_groups = array_filter([$sticky_threads_item_group, $regular_threads_item_group]);
        if ($item_groups === []) {
            $vc_container = $this->factory->panel()->listing()->standard(
                $this->lng->txt('thread_overview'),
                [$this->factory->item()->group($this->lng->txt('frm_no_threads'), [])]
            );
        } else {
            $vc_container = $this->factory->panel()->listing()->standard(
                $this->lng->txt('thread_overview'),
                $item_groups
            )->withViewControls($view_controls);
        }

        $default_html = $this->renderer->render($vc_container);
        $modals = $this->renderer->render($this->modal_collection);

        $this->initStyleSheets();

        $forwarder = new ilForumPageCommandForwarder(
            $GLOBALS['DIC']['http'],
            $this->ctrl,
            $this->tabs_gui,
            $this->lng,
            $this->object,
            $this->user,
            $this->content_style_domain
        );
        $forwarder->setPresentationMode(ilForumPageCommandForwarder::PRESENTATION_MODE_PRESENTATION);

        $this->tpl->setContent($forwarder->forward() . $default_html . $modals);
    }

    private function getEffectiveThreadSortation(): ThreadSortation
    {
        $requested_sortation = $this->http->wrapper()->query()->retrieve(
            'thread_sortation',
            $this->refinery->byTrying([
                $this->refinery->kindlyTo()->int(),
                $this->refinery->always(null)
            ])
        );

        $sortation = ThreadSortation::tryFrom(
            $requested_sortation ?? ThreadSortation::DEFAULT_SORTATION->value
        );
        if ($sortation === null) {
            $sortation = ThreadSortation::DEFAULT_SORTATION;
        }

        return $sortation;
    }

    private function getRequestedThreadOffset(): int
    {
        return $this->http->wrapper()->query()->retrieve(
            'page',
            $this->refinery->byTrying([
                $this->refinery->kindlyTo()->int(),
                $this->refinery->always(0)
            ])
        );
    }

    private function initializeThreadOffsetAndLimit(ilForumThreadObjectTableGUI $tbl): void
    {
        $limit = ilForumProperties::PAGE_SIZE_THREAD_OVERVIEW;
        $offset = $this->getRequestedThreadOffset() * $limit;

        $tbl->setOffset($offset);
        $tbl->setLimit($limit);
    }

    private function initializeThreadSortation(ilForumThreadObjectTableGUI $tbl): void
    {
        $sortation = $this->getEffectiveThreadSortation();
        $tbl->setOrderDirection($sortation->direction());
        $tbl->setOrderField($sortation->field());
    }

    private function getSortationViewControl(): \ILIAS\UI\Component\ViewControl\Sortation
    {
        $offset = $this->getRequestedThreadOffset();
        if ($offset > 0) {
            $this->ctrl->setParameter($this, 'page', $offset);
        }

        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
        $base_url = $this->ctrl->getLinkTarget($this, 'showThreads');

        $translation_keys = [];
        foreach (ThreadSortation::cases() as $sortation) {
            $translation_keys[$sortation->value] = $this->lng->txt($sortation->languageId());
        }
        $this->ctrl->clearParameters($this);

        return $this->factory->viewControl()->sortation(
            $translation_keys,
            (string) $this->getEffectiveThreadSortation()->value
        )->withTargetURL($base_url, 'thread_sortation');
    }

    /**
     * @return list<int>
     */
    private function retrieveThreadIds(): array
    {
        $thread_ids = [];
        $query = $this->http->wrapper()->query()->has('thr_pk');
        $post = $this->http->wrapper()->post()->has('thread_ids');
        if ($post) {
            $thread_ids = $this->http->wrapper()->post()->retrieve(
                'thread_ids',
                $this->refinery->kindlyTo()->listOf($this->refinery->kindlyTo()->int())
            );
        } elseif ($query) {
            $thread_ids = $this->http->wrapper()->query()->retrieve(
                'thr_pk',
                $this->refinery->kindlyTo()->listOf($this->refinery->kindlyTo()->int())
            );
        }

        return $thread_ids;
    }

    /**
     * @return array <string, string>
     */
    private function getThreadProperties(ilForumTopic $forum_topic): array
    {
        $this->ctrl->setParameter($this, 'thr_pk', $forum_topic->getId());
        $unread_counter = $forum_topic->getNumUnreadPosts();
        $is_top_thread = $forum_topic->isSticky();

        if ($unread_counter === 0) {
            $unread = '';
        } else {
            $unread = ' (' . $unread_counter . ' ' . $this->lng->txt('unread') . ')';
        }

        $authorinfo = new ilForumAuthorInformation(
            $forum_topic->getThrAuthorId(),
            $forum_topic->getDisplayUserId(),
            (string) $forum_topic->getUserAlias(),
            (string) $forum_topic->getImportName()
        );

        $properties = [
            $this->lng->txt('forums_created_by') => $authorinfo->getAuthorName(),
            $this->lng->txt('forums_articles') => $forum_topic->getNumPosts() . $unread,
            $this->lng->txt('forums_last_post') => $this->object->Forum->convertDate($forum_topic->getLastPost()->getCreateDate()),
            $this->lng->txt('visits') => $forum_topic->getVisits(),
        ];

        if ($is_top_thread === true) {
            $is_top_thread = [$this->lng->txt('relevance') => $this->lng->txt('sticky')];
            $properties = array_merge($properties, $is_top_thread);
        }

        $rating_property = [];
        if ($this->objProperties->isIsThreadRatingEnabled()) {
            $rating = new ilRatingGUI();
            $rating->setObject(
                $this->object->getId(),
                $this->object->getType(),
                $forum_topic->getId(),
                'thread'
            );
            $rating->setUserId($this->user->getId());
            $rating_property = [$this->lng->txt('frm_rating') => $rating->getHTML()];
        }
        $this->ctrl->setParameter($this, 'thr_pk', null);

        return array_merge($properties, $rating_property);
    }

    private function getLinkActionForThread(
        int $ref_id,
        string $title,
        string $cmd,
        int $thread_id = null
    ): \ILIAS\UI\Component\Link\Link {
        $this->ctrl->setParameter($this, 'ref_id', $ref_id);
        $this->ctrl->setParameter($this, 'thr_pk', $thread_id);
        $this->ctrl->setParameter($this, 'page', 0);
        $url = $this->ctrl->getLinkTarget($this, $cmd);
        $this->ctrl->setParameter($this, 'ref_id', null);
        $this->ctrl->setParameter($this, 'thr_pk', null);
        $this->ctrl->setParameter($this, 'page', null);

        return $this->uiFactory->link()->standard($title, $url);
    }

    private function getActionsForThreadOverview(int $ref_id, ilForumTopic $forum_topic): Standard
    {
        $actions = $this->uiFactory->dropdown()->standard([]);
        if ($this->is_moderator) {
            $open_close = $this->getOpenCloseActionForThread($forum_topic, $ref_id);
            $stick_or_no_stick = $this->getStickyActionForThread($forum_topic, $ref_id);
            $edit_title_modal = $this->getEditTitleModal($forum_topic);
            $this->modal_collection[] = $edit_title_modal;
            $edit_title = $this->factory->button()->shy($this->lng->txt('frm_edit_title'), '#')->withOnClick(
                $edit_title_modal->getShowSignal()
            );

            $move = $this->getLinkActionForThread(
                $ref_id,
                $this->lng->txt('move_thread_to_forum'),
                'move',
                $forum_topic->getId()
            );
            $merge = $this->getLinkActionForThread(
                $ref_id,
                $this->lng->txt('merge_posts_into_thread'),
                'mergeThreads',
                $forum_topic->getId()
            );
            $delete = $this->getLinkActionForThread(
                $ref_id,
                $this->lng->txt('delete_thread'),
                'confirmDeleteThreads',
                $forum_topic->getId()
            );
            $actions = $this->uiFactory->dropdown()->standard([
                $stick_or_no_stick,
                $open_close,
                $edit_title,
                $move,
                $merge,
                $delete
            ]);
        }

        return $actions;
    }

    private function getEditTitleModal(ilForumTopic $topic): RoundTrip
    {
        $topic_id = $this->refinery->kindlyTo()->string()->transform($topic->getId());
        $this->ctrl->setParameter($this, 'thr_pk', (string) $topic_id);
        $target = $this->ctrl->getLinkTargetByClass(
            ilObjForumGUI::class,
            'editThread'
        );
        $this->ctrl->setParameter($this, 'thread_id', null);

        return $this->factory->modal()->roundtrip(
            $this->lng->txt('frm_edit_title'),
            [],
            [
                $this->factory->input()->field()->text($this->lng->txt('frm_edit_title'))->withValue($topic->getSubject()),
            ],
            $target
        );
    }

    private function addDraftButtonIfDraftsExists(string $cmd): void
    {
        $drafts = ilForumPostDraft::getThreadDraftData(
            $this->user->getId(),
            ilObjForum::lookupForumIdByObjId($this->object->getId())
        );
        if ($drafts === []) {
            return;
        }

        $table = new ForumDraftsTable(
            $this->object,
            $this->ui_factory,
            $this->httpRequest,
            $this->lng,
            $cmd,
            $this->ctrl,
            new DataFactory(),
            $this->user,
            $this->access->checkAccess('add_thread', '', $this->object->getRefId()),
            $this
        );
        $threadsTemplate = new ilTemplate(
            'tpl.forums_threads_liste.html',
            true,
            true,
            'components/ILIAS/Forum'
        );
        $threadsTemplate->setVariable('THREADS_DRAFTS_TABLE', $this->uiRenderer->render($table->getComponent()));

        $draft_modal = $this->factory->modal()->roundtrip(
            $this->lng->txt('drafts'),
            [$this->factory->legacy($threadsTemplate->get())]
        );
        $this->modal_collection[] = $draft_modal;
        $edit_title = $this->factory->button()->standard($this->lng->txt('drafts'), '#')->withOnClick(
            $draft_modal->getShowSignal()
        );
        $this->toolbar->addComponent($edit_title);
    }

    private function getStickyActionForThread(ilForumTopic $forum_topic, int $ref_id): \ILIAS\UI\Component\Link\Link
    {
        return $this->getLinkActionForThread(
            $ref_id,
            $this->lng->txt($forum_topic->isSticky() ? 'make_topics_non_sticky' : 'make_topics_sticky'),
            $forum_topic->isSticky() ? 'make_topics_non_sticky' : 'makesticky',
            $forum_topic->getId()
        );
    }

    private function getOpenCloseActionForThread(ilForumTopic $forum_topic, int $ref_id): \ILIAS\UI\Component\Link\Link
    {
        return $this->getLinkActionForThread(
            $ref_id,
            $this->lng->txt($forum_topic->isClosed() ? 'reopen_topics' : 'close_topics'),
            $forum_topic->isClosed() ? 'reopen' : 'close',
            $forum_topic->getId()
        );
    }

    private function markTopThreadInOverview(
        ilForumTopic $current_thread,
        Item $list_item
    ): Item {
        if ($current_thread->isSticky()) {
            $df = new \ILIAS\Data\Factory();
            $list_item = $list_item->withColor($df->color('#B54F00'));
        }

        return $list_item;
    }

    private function initStyleSheets(): void
    {
        $this->content_style_gui->addCss($this->tpl, $this->ref_id);
        $this->tpl->setCurrentBlock('SyntaxStyle');
        $this->tpl->setVariable('LOCATION_SYNTAX_STYLESHEET', ilObjStyleSheet::getSyntaxStylePath());
        $this->tpl->parseCurrentBlock();
    }

    private function editStylePropertiesObject(): void
    {
        $this->content_style_gui->redirectToObjectSettings();
    }

    /**
     * @param list<ilForumPostDraft> $drafts
     */
    private function renderDraftContent(
        ilTemplate $tpl,
        string $action,
        ilForumPost $referencePosting,
        array $drafts
    ): void {
        $frm = $this->object->Forum;

        $draft_id = $this->retrieveDraftId();
        foreach ($drafts as $draft) {
            $tmp_file_obj = new ilFileDataForumDrafts($this->object->getId(), $draft->getDraftId());
            $filesOfDraft = $tmp_file_obj->getFilesOfPost();
            ksort($filesOfDraft);

            if ($action !== 'showdraft' && $filesOfDraft !== []) {
                foreach ($filesOfDraft as $file) {
                    $tpl->setCurrentBlock('attachment_download_row');
                    $this->ctrl->setParameter($this, 'draft_id', $draft->getDraftId());
                    $this->ctrl->setParameter($this, 'file', $file['md5']);
                    $tpl->setVariable('HREF_DOWNLOAD', $this->ctrl->getLinkTarget($this, 'viewThread'));
                    $tpl->setVariable('TXT_FILENAME', $file['name']);
                    $this->ctrl->setParameter($this, 'file', '');
                    $this->ctrl->setParameter($this, 'draft_id', '');
                    $this->ctrl->clearParameters($this);
                    $tpl->parseCurrentBlock();
                }

                $tpl->setCurrentBlock('attachments');
                $tpl->setVariable('TXT_ATTACHMENTS_DOWNLOAD', $this->lng->txt('forums_attachments'));
                $tpl->setVariable(
                    'DOWNLOAD_IMG',
                    $this->uiRenderer->render(
                        $this->uiFactory->symbol()->glyph()->attachment($this->lng->txt('forums_download_attachment'))
                    )
                );
                if (count($filesOfDraft) > 1) {
                    $this->ctrl->setParameter($this, 'draft_id', $draft->getDraftId());
                    $download_zip_button = $this->uiFactory->button()
                                                           ->standard(
                                                               $this->lng->txt('download'),
                                                               $this->ctrl->getLinkTarget($this, 'deliverDraftZipFile')
                                                           );
                    $this->ctrl->setParameter($this, 'draft_id', '');
                    $tpl->setVariable('DOWNLOAD_ZIP', $this->uiRenderer->render($download_zip_button));
                }
                $tpl->parseCurrentBlock();
            }

            $page = $this->http->wrapper()->query()->retrieve(
                'page',
                $this->refinery->byTrying([$this->refinery->kindlyTo()->int(), $this->refinery->always(0)])
            );

            $this->renderSplitButton(
                $tpl,
                $action,
                false,
                $referencePosting,
                (int) $page,
                $draft
            );

            $rowCol = 'tblrowmarked';
            $tpl->setVariable('ROWCOL', ' ' . $rowCol);
            $depth = $referencePosting->getDepth() - 1;
            if ($this->selectedSorting === ilForumProperties::VIEW_TREE) {
                ++$depth;
            }
            $tpl->setVariable('DEPTH', $depth);

            $this->ctrl->setParameter($this, 'pos_pk', $referencePosting->getId());
            $this->ctrl->setParameter($this, 'thr_pk', $referencePosting->getThreadId());
            $this->ctrl->setParameter($this, 'draft_id', $draft->getDraftId());

            $backurl = urlencode($this->ctrl->getLinkTarget($this, 'viewThread', (string) $referencePosting->getId()));

            $this->ctrl->setParameter($this, 'backurl', $backurl);
            $this->ctrl->setParameter($this, 'thr_pk', $referencePosting->getThreadId());
            $this->ctrl->setParameter($this, 'user', $draft->getPostDisplayUserId());

            $authorinfo = new ilForumAuthorInformation(
                $draft->getPostAuthorId(),
                $draft->getPostDisplayUserId(),
                $draft->getPostUserAlias(),
                '',
                [
                    'href' => $this->ctrl->getLinkTarget($this, 'showUser')
                ]
            );

            $this->ctrl->clearParameters($this);

            if ($authorinfo->hasSuffix()) {
                $tpl->setVariable('AUTHOR', $authorinfo->getSuffix());
                $tpl->setVariable('USR_NAME', $draft->getPostUserAlias());
            } else {
                $tpl->setVariable('AUTHOR', $authorinfo->getLinkedAuthorShortName());
                if ($authorinfo->getAuthorName(true) && !$this->objProperties->isAnonymized()) {
                    $tpl->setVariable('USR_NAME', $authorinfo->getAuthorName(true));
                }
            }
            $tpl->setVariable('DRAFT_ANCHOR', 'draft_' . $draft->getDraftId());

            $tpl->setVariable('USR_IMAGE', $authorinfo->getProfilePicture());
            $tpl->setVariable(
                'USR_ICON_ALT',
                ilLegacyFormElementsUtil::prepareFormOutput($authorinfo->getAuthorShortName())
            );
            if ($authorinfo->getAuthor()->getId() && ilForum::_isModerator(
                $this->object->getRefId(),
                $draft->getPostAuthorId()
            )) {
                if ($authorinfo->getAuthor()->getGender() === 'f') {
                    $tpl->setVariable('ROLE', $this->lng->txt('frm_moderator_f'));
                } elseif ($authorinfo->getAuthor()->getGender() === 'm') {
                    $tpl->setVariable('ROLE', $this->lng->txt('frm_moderator_m'));
                } elseif ($authorinfo->getAuthor()->getGender() === 'n') {
                    $tpl->setVariable('ROLE', $this->lng->txt('frm_moderator_n'));
                }
            }

            if ($draft->getUpdateUserId() > 0) {
                $draft->setPostUpdate($draft->getPostUpdate());

                $this->ctrl->setParameter($this, 'backurl', $backurl);
                $this->ctrl->setParameter($this, 'thr_pk', $referencePosting->getThreadId());
                $this->ctrl->setParameter($this, 'user', $referencePosting->getUpdateUserId());
                $this->ctrl->setParameter($this, 'draft_id', $draft->getDraftId());

                $authorinfo = new ilForumAuthorInformation(
                    $draft->getPostAuthorId(),
                    // We assume the editor is the author here
                    $draft->getPostDisplayUserId(),
                    $draft->getPostUserAlias(),
                    '',
                    ['href' => $this->ctrl->getLinkTarget($this, 'showUser')]
                );

                $this->ctrl->clearParameters($this);

                $tpl->setVariable(
                    'POST_UPDATE_TXT',
                    $this->lng->txt('edited_on') . ': ' . $frm->convertDate($draft->getPostUpdate()) . ' - ' . strtolower($this->lng->txt('by'))
                );
                $tpl->setVariable('UPDATE_AUTHOR', $authorinfo->getLinkedAuthorShortName());
                if ($authorinfo->getAuthorName(true) && !$this->objProperties->isAnonymized() && !$authorinfo->hasSuffix()) {
                    $tpl->setVariable('UPDATE_USR_NAME', $authorinfo->getAuthorName(true));
                }
            }

            $draft->setPostMessage($frm->prepareText($draft->getPostMessage()));

            $tpl->setVariable('SUBJECT', $draft->getPostSubject());
            $tpl->setVariable('POST_DATE', $frm->convertDate($draft->getPostDate()));

            $tpl->setVariable('POST_DRAFT_TEXT', $this->lng->txt('post_draft_info'));

            if (!$referencePosting->isCensored() || ($this->objCurrentPost->getId() === $referencePosting->getId() && $action === 'censor')) {
                $spanClass = '';
                if (ilForum::_isModerator($this->ref_id, $draft->getPostDisplayUserId())) {
                    $spanClass = 'moderator';
                }

                if ($draft->getPostMessage() === strip_tags($draft->getPostMessage())) {
                    // We can be sure, that there are not html tags
                    $draft->setPostMessage(nl2br($draft->getPostMessage()));
                }

                if ($spanClass !== '') {
                    $tpl->setVariable(
                        'POST',
                        '<span class="' . $spanClass . '">' . ilRTE::_replaceMediaObjectImageSrc(
                            $draft->getPostMessage(),
                            1
                        ) . '</span>'
                    );
                } else {
                    $tpl->setVariable('POST', ilRTE::_replaceMediaObjectImageSrc($draft->getPostMessage(), 1));
                }
            }

            if ($action === 'editdraft' && $draft->getDraftId() === $draft_id) {
                $oEditReplyForm = $this->getReplyEditForm();

                if (!$this->objCurrentTopic->isClosed() && in_array($this->requestAction, ['showdraft', 'editdraft'])) {
                    $this->renderPostingForm($tpl, $frm, $referencePosting, $this->requestAction);
                }

                $tpl->setVariable('EDIT_DRAFT_ANCHOR', 'draft_edit_' . $draft->getDraftId());
                $tpl->setVariable('DRAFT_FORM', $oEditReplyForm->getHTML() . $this->modal_history);
            }

            $tpl->parseCurrentBlock();
        }
    }

    protected function renderPostContent(
        ilTemplate $tpl,
        ilForumPost $node,
        string $action,
        int $pageIndex,
        int $postIndex
    ): void {
        $forumObj = $this->object;
        $frm = $this->object->Forum;

        $fileDataOfForum = new ilFileDataForum($forumObj->getId(), $node->getId());

        $filesOfPost = $fileDataOfForum->getFilesOfPost();
        ksort($filesOfPost);
        if ($filesOfPost !== [] && ($action !== 'showedit' || $node->getId() !== $this->objCurrentPost->getId())) {
            foreach ($filesOfPost as $file) {
                $tpl->setCurrentBlock('attachment_download_row');
                $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
                $this->ctrl->setParameter($this, 'file', $file['md5']);
                $tpl->setVariable('HREF_DOWNLOAD', $this->ctrl->getLinkTarget($this, 'viewThread'));
                $tpl->setVariable('TXT_FILENAME', $file['name']);
                $this->ctrl->clearParameters($this);
                $tpl->parseCurrentBlock();
            }
            $tpl->setCurrentBlock('attachments');
            $tpl->setVariable('TXT_ATTACHMENTS_DOWNLOAD', $this->lng->txt('forums_attachments'));
            $tpl->setVariable(
                'DOWNLOAD_IMG',
                $this->uiRenderer->render(
                    $this->uiFactory->symbol()->glyph()->attachment($this->lng->txt('forums_download_attachment'))
                )
            );
            if (count($filesOfPost) > 1) {
                $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
                $download_zip_button = $this->uiFactory->button()
                                                       ->standard(
                                                           $this->lng->txt('download'),
                                                           $this->ctrl->getLinkTarget($this, 'deliverZipFile')
                                                       );
                $tpl->setVariable('DOWNLOAD_ZIP', $this->uiRenderer->render($download_zip_button));
            }
            $tpl->parseCurrentBlock();
        }
        $this->renderSplitButton($tpl, $action, true, $node, $pageIndex);

        $tpl->setVariable('POST_ANKER', $node->getId());
        $tpl->setVariable('TXT_PERMA_LINK', $this->lng->txt('perma_link'));
        $tpl->setVariable('PERMA_TARGET', '_top');

        $rowCol = ilUtil::switchColor($postIndex, 'tblrow1', 'tblrow2');
        if (($this->is_moderator || $node->isOwner($this->user->getId())) && !$node->isActivated() && !$this->objCurrentTopic->isClosed()) {
            $rowCol = 'ilPostingNeedsActivation';
        } elseif ($this->objProperties->getMarkModeratorPosts()) {
            $isAuthorModerator = ilForum::_isModerator($this->object->getRefId(), $node->getPosAuthorId());
            if ($isAuthorModerator && $node->isAuthorModerator() === null) {
                $rowCol = 'ilModeratorPosting';
            } elseif ($node->isAuthorModerator()) {
                $rowCol = 'ilModeratorPosting';
            }
        }

        if (
            (!in_array($action, ['delete', 'censor']) && !$this->displayConfirmPostActivation()) ||
            $this->objCurrentPost->getId() !== $node->getId()
        ) {
            $tpl->setVariable('ROWCOL', ' ' . $rowCol);
        } else {
            $rowCol = 'tblrowmarked';
        }

        if ($node->isCensored()) {
            if ($action !== 'censor') {
                $tpl->setVariable('TXT_CENSORSHIP_ADVICE', $this->lng->txt('post_censored_comment_by_moderator'));
            }

            $rowCol = 'tblrowmarked';
        }

        $tpl->setVariable('ROWCOL', ' ' . $rowCol);
        $tpl->setVariable('DEPTH', $node->getDepth() - 1);
        if (!$node->isActivated() && ($node->isOwner($this->user->getId()) || $this->is_moderator)) {
            $tpl->setVariable('POST_NOT_ACTIVATED_YET', $this->lng->txt('frm_post_not_activated_yet'));
        }

        $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
        $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
        $backurl = urlencode($this->ctrl->getLinkTarget($this, 'viewThread', (string) $node->getId()));
        $this->ctrl->clearParameters($this);

        $this->ctrl->setParameter($this, 'backurl', $backurl);
        $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
        $this->ctrl->setParameter($this, 'user', $node->getDisplayUserId());
        $authorinfo = new ilForumAuthorInformation(
            $node->getPosAuthorId(),
            $node->getDisplayUserId(),
            (string) $node->getUserAlias(),
            (string) $node->getImportName(),
            [
                'href' => $this->ctrl->getLinkTarget($this, 'showUser')
            ]
        );
        $this->ctrl->clearParameters($this);

        if ($authorinfo->hasSuffix()) {
            if (!$authorinfo->isDeleted()) {
                $tpl->setVariable('USR_NAME', $authorinfo->getAlias());
            }
            $tpl->setVariable('AUTHOR', $authorinfo->getSuffix());
        } else {
            if ($authorinfo->getAuthorName(true) && !$this->objProperties->isAnonymized()) {
                $tpl->setVariable('USR_NAME', $authorinfo->getAuthorName(true));
            }
            $tpl->setVariable('AUTHOR', $authorinfo->getLinkedAuthorShortName());
        }

        $tpl->setVariable('USR_IMAGE', $authorinfo->getProfilePicture());
        $tpl->setVariable(
            'USR_ICON_ALT',
            ilLegacyFormElementsUtil::prepareFormOutput($authorinfo->getAuthorShortName())
        );
        $isModerator = ilForum::_isModerator($this->ref_id, $node->getPosAuthorId());
        if ($isModerator && $authorinfo->getAuthor()->getId()) {
            $authorRole = $this->lng->txt('frm_moderator_n');
            if (is_string($authorinfo->getAuthor()->getGender()) && $authorinfo->getAuthor()->getGender() !== '') {
                $authorRole = $this->lng->txt('frm_moderator_' . $authorinfo->getAuthor()->getGender());
            }
            $tpl->setVariable('ROLE', $authorRole);
        }

        if ($node->getUpdateUserId() > 0) {
            $node->setChangeDate($node->getChangeDate());

            $this->ctrl->setParameter($this, 'backurl', $backurl);
            $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
            $this->ctrl->setParameter($this, 'user', $node->getUpdateUserId());
            $update_user_id = $node->getUpdateUserId();
            if ($node->getDisplayUserId() === 0 && $node->getPosAuthorId() === $node->getUpdateUserId()) {
                $update_user_id = $node->getDisplayUserId();
            }
            $authorinfo = new ilForumAuthorInformation(
                $node->getPosAuthorId(),
                $update_user_id,
                (string) $node->getUserAlias(),
                (string) $node->getImportName(),
                [
                    'href' => $this->ctrl->getLinkTarget($this, 'showUser')
                ]
            );
            $this->ctrl->clearParameters($this);

            $tpl->setVariable(
                'POST_UPDATE_TXT',
                $this->lng->txt('edited_on') . ': ' . $frm->convertDate($node->getChangeDate()) . ' - ' . strtolower($this->lng->txt('by'))
            );
            $tpl->setVariable('UPDATE_AUTHOR', $authorinfo->getLinkedAuthorShortName());
            if ($authorinfo->getAuthorName(true) && !$this->objProperties->isAnonymized() && !$authorinfo->hasSuffix()) {
                $tpl->setVariable('UPDATE_USR_NAME', $authorinfo->getAuthorName(true));
            }
        }

        if ($this->selectedSorting === ilForumProperties::VIEW_TREE
            && $node->getId() !== $this->selected_post_storage->get($node->getThreadId())) {
            $target = $this->uiFactory->symbol()->icon()->custom(
                ilUtil::getImagePath('object/target.svg'),
                $this->lng->txt('target_select')
            );

            $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
            $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());

            $tpl->setVariable(
                'TARGET',
                $this->uiRenderer->render(
                    $this->uiFactory->link()->bulky(
                        $target,
                        $this->lng->txt('select'),
                        new \ILIAS\Data\URI(
                            ILIAS_HTTP_PATH . '/' . $this->ctrl->getLinkTarget($this, 'selectPost', (string) $node->getId())
                        )
                    )
                )
            );
        }

        $node->setMessage($frm->prepareText($node->getMessage()));

        if ($this->user->isAnonymous() || $node->isPostRead()) {
            $tpl->setVariable('SUBJECT', $node->getSubject());
        } else {
            $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
            $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
            $this->ctrl->setParameter($this, 'page', $pageIndex);
            $this->ctrl->setParameter(
                $this,
                'orderby',
                $this->getOrderByParam()
            );
            $this->ctrl->setParameter($this, 'viewmode', $this->selectedSorting);
            $mark_post_target = $this->ctrl->getLinkTarget($this, 'markPostRead', (string) $node->getId());

            $tpl->setVariable(
                'SUBJECT',
                '<a href="' . $mark_post_target . '"><b>' . $node->getSubject() . '</b></a>'
            );
        }

        $tpl->setVariable('POST_DATE', $frm->convertDate($node->getCreateDate()));

        if (!$node->isCensored() || ($this->objCurrentPost->getId() === $node->getId() && $action === 'censor')) {
            $spanClass = '';
            if (ilForum::_isModerator($this->ref_id, $node->getDisplayUserId())) {
                $spanClass = 'moderator';
            }

            // possible bugfix for mantis #8223
            if ($node->getMessage() === strip_tags($node->getMessage())) {
                // We can be sure, that there are not html tags
                $node->setMessage(nl2br($node->getMessage()));
            }

            if ($spanClass !== '') {
                $tpl->setVariable(
                    'POST',
                    '<span class="' . $spanClass . '">' .
                    ilRTE::_replaceMediaObjectImageSrc($node->getMessage(), 1) .
                    '</span>'
                );
            } else {
                $tpl->setVariable('POST', ilRTE::_replaceMediaObjectImageSrc($node->getMessage(), 1));
            }
        } else {
            $tpl->setVariable(
                'POST',
                '<span class="moderator">' . nl2br((string) $node->getCensorshipComment()) . '</span>'
            );
        }

        $tpl->parseCurrentBlock();
    }

    protected function selectPostObject(): void
    {
        $thr_pk = (int) $this->httpRequest->getQueryParams()['thr_pk'];
        $pos_pk = (int) $this->httpRequest->getQueryParams()['pos_pk'];

        $this->selected_post_storage->set(
            $thr_pk,
            $pos_pk
        );

        $this->viewThreadObject();
    }

    /**
     * @param ilObject|ilObjForum $new_object
     */
    protected function afterSave(ilObject $new_object): void
    {
        $this->tpl->setOnScreenMessage('success', $this->lng->txt('frm_added'), true);
        $this->ctrl->setParameter($this, 'ref_id', $new_object->getRefId());
        $this->ctrl->redirect($this, 'createThread');
    }

    protected function getTabs(): void
    {
        if ($this->in_page_editor_style_context) {
            return;
        }

        $this->ilHelp->setScreenIdComponent('frm');

        $this->ctrl->setParameter($this, 'ref_id', $this->ref_id);

        $active = [
            '',
            'showThreads',
            'view',
            'markAllRead',
            'enableForumNotification',
            'disableForumNotification',
            'moveThreads',
            'performMoveThreads',
            'cancelMoveThreads',
            'performThreadsAction',
            'createThread',
            'addThread',
            'showUser',
            'confirmDeleteThreads',
            'merge',
            'mergeThreads',
            'performMergeThreads'
        ];

        $force_active = false;
        if (in_array($this->ctrl->getCmd(), $active, true)) {
            $force_active = true;
        }

        if ($this->access->checkAccess(
            'read',
            '',
            $this->ref_id
        )) {
            $this->tabs_gui->addTarget(
                self::UI_TAB_ID_THREADS,
                $this->ctrl->getLinkTarget($this, 'showThreads'),
                $this->ctrl->getCmd(),
                static::class,
                '',
                $force_active
            );
        }

        if ($this->access->checkAccess('visible', '', $this->ref_id) || $this->access->checkAccess(
            'read',
            '',
            $this->ref_id
        )) {
            $cmdClass = $this->http->wrapper()->query()->retrieve(
                'cmdClass',
                $this->refinery->byTrying([$this->refinery->kindlyTo()->string(), $this->refinery->always('')])
            );

            $force_active = $this->ctrl->getNextClass() === 'ilinfoscreengui' || strtolower($cmdClass) === 'ilnotegui';
            $this->tabs_gui->addTarget(
                self::UI_TAB_ID_INFO,
                $this->ctrl->getLinkTargetByClass([self::class, ilInfoScreenGUI::class], 'showSummary'),
                ['showSummary', 'infoScreen'],
                '',
                '',
                $force_active
            );
        }

        if ($this->access->checkAccess('write', '', $this->ref_id)) {
            $force_active = $this->ctrl->getCmd() === 'edit';
            $this->tabs_gui->addTarget(
                self::UI_TAB_ID_SETTINGS,
                $this->ctrl->getLinkTarget($this, 'edit'),
                'edit',
                $this::class,
                '',
                $force_active
            );
        }

        if ($this->access->checkAccess('write', '', $this->ref_id)) {
            $this->tabs_gui->addTarget(
                self::UI_TAB_ID_MODERATORS,
                $this->ctrl->getLinkTargetByClass(ilForumModeratorsGUI::class, 'showModerators'),
                'showModerators',
                static::class
            );
        }

        if (ilLearningProgressAccess::checkAccess($this->object->getRefId())) {
            $this->tabs_gui->addTab(
                'learning_progress',
                $this->lng->txt('learning_progress'),
                $this->ctrl->getLinkTargetByClass(ilLearningProgressGUI::class)
            );
        }

        if ($this->settings->get('enable_fora_statistics', '0')) {
            $hasStatisticsAccess = $this->access->checkAccess('write', '', $this->ref_id);
            if (!$hasStatisticsAccess) {
                $hasStatisticsAccess = (
                    $this->objProperties->isStatisticEnabled() &&
                    $this->access->checkAccess('read', '', $this->ref_id)
                );
            }

            if ($hasStatisticsAccess) {
                $force_active = $this->ctrl->getCmd() === 'showStatistics';
                $this->tabs_gui->addTarget(
                    self::UI_TAB_ID_STATS,
                    $this->ctrl->getLinkTarget($this, 'showStatistics'),
                    'showStatistics',
                    static::class,
                    '',
                    $force_active
                );
            }
        }

        if ($this->access->checkAccess('write', '', $this->object->getRefId())) {
            $this->tabs_gui->addTarget(
                self::UI_TAB_ID_EXPORT,
                $this->ctrl->getLinkTargetByClass(ilExportGUI::class, ''),
                '',
                'ilexportgui'
            );
        }

        if ($this->access->checkAccess('edit_permission', '', $this->ref_id)) {
            $this->tabs_gui->addTarget(
                self::UI_TAB_ID_PERMISSIONS,
                $this->ctrl->getLinkTargetByClass([static::class, ilPermissionGUI::class], 'perm'),
                ['perm', 'info', 'owner'],
                'ilpermissiongui'
            );
        }
    }

    public function showStatisticsObject(): void
    {
        if (!$this->settings->get('enable_fora_statistics', '0')) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->objProperties->isStatisticEnabled()) {
            if ($this->access->checkAccess('write', '', $this->object->getRefId())) {
                $this->tpl->setOnScreenMessage('info', $this->lng->txt('frm_statistics_disabled_for_participants'));
            } else {
                $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
            }
        }

        $this->object->Forum->setForumId($this->object->getId());

        $tbl = new \ILIAS\Forum\Statistics\ForumStatisticsTable(
            $this->object,
            $this->objProperties,
            ilLearningProgressAccess::checkAccess($this->object->getRefId()),
            $this->access->checkRbacOrPositionPermissionAccess(
                'read_learning_progress',
                'read_learning_progress',
                $this->object->getRefId()
            ),
            $this->user,
            $this->ui_factory,
            $this->request,
            $this->lng
        );
        $this->tpl->setContent($this->uiRenderer->render($tbl->getComponent()));
    }

    public static function _goto($a_target, $a_thread = 0, $a_posting = 0): void
    {
        global $DIC;
        $main_tpl = $DIC->ui()->mainTemplate();

        $ilAccess = $DIC->access();
        $lng = $DIC->language();
        $ilErr = $DIC['ilErr'];

        $a_target = is_numeric($a_target) ? (int) $a_target : 0;
        $a_thread = is_numeric($a_thread) ? (int) $a_thread : 0;
        if ($ilAccess->checkAccess('read', '', $a_target)) {
            if ($a_thread !== 0) {
                $objTopic = new ilForumTopic($a_thread);
                if ($objTopic->getFrmObjId() &&
                    $objTopic->getFrmObjId() !== ilObject::_lookupObjectId($a_target)) {
                    $ref_ids = ilObject::_getAllReferences($objTopic->getFrmObjId());
                    foreach ($ref_ids as $ref_id) {
                        if ($ilAccess->checkAccess('read', '', $ref_id)) {
                            $new_ref_id = $ref_id;
                            break;
                        }
                    }

                    if (isset($new_ref_id) && $new_ref_id !== $a_target) {
                        $DIC->ctrl()->redirectToURL(
                            ILIAS_HTTP_PATH . '/goto.php?target=frm_' . $new_ref_id . '_' . $a_thread . '_' . $a_posting
                        );
                    }
                }

                $DIC->ctrl()->setParameterByClass(self::class, 'ref_id', (string) ((int) $a_target));
                if (is_numeric($a_thread)) {
                    $DIC->ctrl()->setParameterByClass(self::class, 'thr_pk', (string) ((int) $a_thread));
                }
                if (is_numeric($a_posting)) {
                    $DIC->ctrl()->setParameterByClass(self::class, 'pos_pk', (string) ((int) $a_posting));
                }
                $DIC->ctrl()->redirectByClass(
                    [ilRepositoryGUI::class, self::class],
                    'viewThread',
                    is_numeric($a_posting) ? (string) ((int) $a_posting) : ''
                );
            } else {
                $DIC->ctrl()->setParameterByClass(self::class, 'ref_id', $a_target);
                $DIC->ctrl()->redirectByClass([ilRepositoryGUI::class, self::class,], '');
                $DIC->http()->close();
            }
        } elseif ($ilAccess->checkAccess('visible', '', $a_target)) {
            $DIC->ctrl()->setParameterByClass(ilInfoScreenGUI::class, 'ref_id', $a_target);
            $DIC->ctrl()->redirectByClass(
                [
                    ilRepositoryGUI::class,
                    self::class,
                    ilInfoScreenGUI::class
                ],
                'showSummary'
            );
        } elseif ($ilAccess->checkAccess('read', '', ROOT_FOLDER_ID)) {
            $main_tpl->setOnScreenMessage('info', sprintf(
                $lng->txt('msg_no_perm_read_item'),
                ilObject::_lookupTitle(ilObject::_lookupObjId($a_target))
            ), true);
            $DIC->http()->close();
        }

        $ilErr->raiseError($lng->txt('msg_no_perm_read'), $ilErr->FATAL);
    }

    public function performDeleteThreadsObject(): void
    {
        $threadIds = $this->retrieveThreadIds();
        if ($threadIds === []) {
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('select_at_least_one_thread'), true);
            $this->ctrl->redirect($this, 'showThreads');
        }

        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $forumObj = new ilObjForum($this->object->getRefId());
        $this->objProperties->setObjId($forumObj->getId());

        $frm = new ilForum();

        $success_message = 'forums_thread_deleted';
        if (count($threadIds) > 1) {
            $success_message = 'forums_threads_deleted';
        }

        $threads = [];
        array_walk($threadIds, function (int $threadId) use (&$threads): void {
            $thread = new ilForumTopic($threadId);
            $this->ensureThreadBelongsToForum($this->object->getId(), $thread);

            $threads[] = $thread;
        });

        $frm->setForumId($forumObj->getId());
        $frm->setForumRefId($forumObj->getRefId());
        foreach ($threads as $thread) {
            $first_node = $frm->getFirstPostNode($thread->getId());
            if (isset($first_node['pos_pk']) && (int) $first_node['pos_pk']) {
                $frm->deletePost((int) $first_node['pos_pk']);
                $this->tpl->setOnScreenMessage('info', $this->lng->txt($success_message), true);
            }
        }
        $this->ctrl->redirect($this, 'showThreads');
    }

    public function confirmDeleteThreadsObject(): void
    {
        $thread_ids = $this->retrieveThreadIds();
        if ($thread_ids === []) {
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('select_at_least_one_thread'));
            $this->ctrl->redirect($this, 'showThreads');
        }

        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        /** @var ilForumTopic[] $threads */
        $threads = [];
        array_walk($thread_ids, function (int $threadId) use (&$threads): void {
            $thread = new ilForumTopic($threadId);
            $this->ensureThreadBelongsToForum($this->object->getId(), $thread);

            $threads[] = $thread;
        });

        $c_gui = new ilConfirmationGUI();

        $c_gui->setFormAction($this->ctrl->getFormAction($this, 'performDeleteThreads'));
        $c_gui->setHeaderText($this->lng->txt('frm_sure_delete_threads'));
        $c_gui->setCancel($this->lng->txt('cancel'), 'showThreads');
        $c_gui->setConfirm($this->lng->txt('confirm'), 'performDeleteThreads');

        foreach ($threads as $thread) {
            $c_gui->addItem('thread_ids[]', (string) $thread->getId(), $thread->getSubject());
        }

        $this->confirmation_gui_html = $c_gui->getHTML();

        $this->hideToolbar(true);
        $this->tpl->setContent($c_gui->getHTML());
    }

    protected function confirmDeleteThreadDraftsObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $draft_ids = $this->http->wrapper()->query()->retrieve(
            'forum_drafts_delete_draft_ids',
            $this->refinery->byTrying([
                $this->refinery->kindlyTo()->listOf($this->refinery->kindlyTo()->int()),
                $this->refinery->always([])
            ])
        );

        if ($draft_ids === []) {
            $draft_ids = $this->http->wrapper()->query()->retrieve(
                'forum_drafts_delete_draft_ids',
                $this->refinery->byTrying([
                    $this->refinery->kindlyTo()->listOf($this->refinery->kindlyTo()->string()),
                    $this->refinery->always([])
                ])
            );

            if (!isset($draft_ids[0]) || $draft_ids[0] !== 'ALL_OBJECTS') {
                $this->tpl->setOnScreenMessage('info', $this->lng->txt('select_at_least_one_thread'));
                $this->showThreadsObject();
                return;
            }

            $draft_ids = array_filter(array_map(
                static function (array $draft): int {
                    return $draft['draft_id'] ?? 0;
                },
                ilForumPostDraft::getThreadDraftData(
                    $this->user->getId(),
                    ilObjForum::lookupForumIdByObjId($this->object->getId())
                )
            ));
        }

        $confirmation = new ilConfirmationGUI();
        $confirmation->setFormAction($this->ctrl->getFormAction($this, 'deleteThreadDrafts'));
        $confirmation->setHeaderText($this->lng->txt('sure_delete_drafts'));
        $confirmation->setCancel($this->lng->txt('cancel'), 'showThreads');
        $confirmation->setConfirm($this->lng->txt('confirm'), 'deleteThreadDrafts');
        $instances = ilForumPostDraft::getDraftInstancesByUserId($this->user->getId());
        foreach ($draft_ids as $draftId) {
            if (array_key_exists($draftId, $instances)) {
                $confirmation->addItem('draft_ids[]', (string) $draftId, $instances[$draftId]->getPostSubject());
            }
        }

        $this->tpl->setContent($confirmation->getHTML());
    }

    public function prepareThreadScreen(ilObjForum $a_forum_obj): void
    {
        $this->ilHelp->setScreenIdComponent('frm');

        $this->tpl->loadStandardTemplate();

        $this->tpl->setTitleIcon(ilObject::_getIcon(0, 'big', 'frm'));

        $this->tabs_gui->setBackTarget(
            $this->lng->txt('frm_all_threads'),
            $this->ctrl->getLinkTarget(
                $this,
                'showThreads'
            )
        );

        /** @var ilForum $frm */
        $frm = $a_forum_obj->Forum;
        $frm->setForumId($a_forum_obj->getId());
    }

    public function performPostActivationObject(): void
    {
        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentPost->getThread());

        $this->objCurrentPost->activatePost();
        $GLOBALS['ilAppEventHandler']->raise(
            'components/ILIAS/Forum',
            'activatedPost',
            [
                'object' => $this->object,
                'ref_id' => $this->object->getRefId(),
                'post' => $this->objCurrentPost
            ]
        );
        $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_post_was_activated'), true);

        $this->viewThreadObject();
    }

    private function deletePostingObject(): void
    {
        if (
            !$this->user->isAnonymous() &&
            !$this->objCurrentTopic->isClosed() && (
                $this->is_moderator ||
                ($this->objCurrentPost->isOwner($this->user->getId()) && !$this->objCurrentPost->hasReplies())
            )
        ) {
            $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentPost->getThread());

            $oForumObjects = $this->getForumObjects();
            $forumObj = $oForumObjects['forumObj'];

            $frm = new ilForum();
            $frm->setForumId($forumObj->getId());
            $frm->setForumRefId($forumObj->getRefId());
            $dead_thr = $frm->deletePost($this->objCurrentPost->getId());

            // if complete thread was deleted ...
            if ($dead_thr === $this->objCurrentTopic->getId()) {
                $frm->setMDB2WhereCondition('top_frm_fk = %s ', ['integer'], [$forumObj->getId()]);
                $topicData = $frm->getOneTopic();
                $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_post_deleted'), true);
                if ($topicData->getTopNumThreads() > 0) {
                    $this->ctrl->redirect($this, 'showThreads');
                } else {
                    $this->ctrl->redirect($this, 'createThread');
                }
            }
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_post_deleted'), true);
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
            $this->ctrl->redirect($this, 'viewThread');
        }

        $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
    }

    private function deletePostingDraftObject(): void
    {
        $this->deleteSelectedDraft();
    }

    private function revokeCensorshipObject(): void
    {
        $this->handleCensorship(true);
    }

    private function addCensorshipObject(): void
    {
        $this->handleCensorship();
    }

    private function getModalActions(): string
    {
        $modalString = '';
        foreach ($this->modalActionsContainer as $modal) {
            $modalString .= $this->uiRenderer->render($modal);
        }

        return $modalString;
    }

    private function handleCensorship(bool $wasRevoked = false): void
    {
        $message = '';
        if ($this->is_moderator && !$this->objCurrentTopic->isClosed()) {
            if ($this->http->wrapper()->post()->has('formData')) {
                $formData = $this->http->wrapper()->post()->retrieve(
                    'formData',
                    $this->refinery->kindlyTo()->listOf($this->refinery->kindlyTo()->string())
                );
                $message = $this->handleFormInput($formData['cens_message']);
            }

            if ($message === '' && $this->http->wrapper()->post()->has('cens_message')) {
                $cens_message = $this->http->wrapper()->post()->retrieve(
                    'cens_message',
                    $this->refinery->kindlyTo()->string()
                );
                $message = $this->handleFormInput($cens_message);
            }
            $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentPost->getThread());

            $oForumObjects = $this->getForumObjects();
            $frm = $oForumObjects['frm'];

            if ($wasRevoked) {
                $frm->postCensorship($this->object, $message, $this->objCurrentPost->getId());
                $this->tpl->setOnScreenMessage('success', $this->lng->txt('frm_censorship_revoked'));
            } else {
                $frm->postCensorship($this->object, $message, $this->objCurrentPost->getId(), 1);
                $this->tpl->setOnScreenMessage('success', $this->lng->txt('frm_censorship_applied'));
            }

            $this->viewThreadObject();
            return;
        }

        $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
    }

    public function askForPostActivationObject(): void
    {
        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $this->setDisplayConfirmPostActivation(true);

        $this->viewThreadObject();
    }

    public function setDisplayConfirmPostActivation(bool $status = false): void
    {
        $this->display_confirm_post_activation = $status;
    }

    public function displayConfirmPostActivation(): bool
    {
        return $this->display_confirm_post_activation;
    }

    protected function toggleThreadNotificationObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentTopic);

        if ($this->objCurrentTopic->isNotificationEnabled($this->user->getId())) {
            $this->objCurrentTopic->disableNotification($this->user->getId());
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_notification_disabled'), true);
        } else {
            $this->objCurrentTopic->enableNotification($this->user->getId());
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_notification_enabled'), true);
        }

        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
        $this->ctrl->redirect($this, 'viewThread');
    }

    protected function toggleStickinessObject(): void
    {
        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentTopic);

        if ($this->objCurrentTopic->isSticky()) {
            $this->objCurrentTopic->unmakeSticky();
        } else {
            $this->objCurrentTopic->makeSticky();
        }

        $this->viewThreadObject();
    }

    public function cancelPostObject(): void
    {
        $this->requestAction = '';
        $draft_id = $this->retrieveDraftId(true);
        if ($draft_id > 0) {
            $draft = ilForumPostDraft::newInstanceByDraftId($draft_id);
            if ($this->hasDraftAccess($draft)) {
                $draft->deleteDraftsByDraftIds([$draft_id]);
            }
        }

        $this->viewThreadObject();
    }

    public function cancelDraftObject(): void
    {
        $this->requestAction = '';
        $draft_id = $this->retrieveDraftId();
        if ($draft_id > 0) {
            $history_entry = new ilForumDraftsHistory();
            $history_entry->populateWithFirstAutosaveByDraftId($draft_id);
            $draft = ilForumPostDraft::newInstanceByDraftId($history_entry->getDraftId());
            if ($this->hasDraftAccess($draft)) {
                $draft->setPostSubject($history_entry->getPostSubject());
                $draft->setPostMessage($history_entry->getPostMessage());

                ilForumUtil::moveMediaObjects(
                    $history_entry->getPostMessage(),
                    ilForumDraftsHistory::MEDIAOBJECT_TYPE,
                    $history_entry->getHistoryId(),
                    ilForumPostDraft::MEDIAOBJECT_TYPE,
                    $draft->getDraftId()
                );

                $draft->updateDraft();

                $history_entry->deleteHistoryByDraftIds([$draft->getDraftId()]);
            }
        }

        $this->ctrl->clearParameters($this);
        $this->viewThreadObject();
    }

    public function addActivationFormModal(ilForumPost $node): ILIAS\UI\Component\Button\Standard
    {
        $form = new ilPropertyFormGUI();
        $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
        $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
        $this->ctrl->setParameter(
            $this,
            'orderby',
            $this->getOrderByParam()
        );
        $form->setFormAction($this->ctrl->getFormAction($this, 'performPostActivation'));
        $this->ctrl->clearParameters($this);
        $form->setId('frm_activate_post_' . $node->getId());
        $form_id = $form->getId();
        $message = $this->uiFactory->messageBox()->confirmation($this->lng->txt('activate_post_txt'));
        $submitBtn = $this->uiFactory->button()->primary(
            $this->lng->txt('activate_only_current'),
            '#'
        )->withOnLoadCode(
            static function (string $id) use ($form_id): string {
                return "
                (function () {
                  const button = document.getElementById('$id');
                  if (!button) return;
                
                  const form = document.getElementById('form_$form_id');
                  if (!form) return;
                
                  button.addEventListener('click', (event) => {
                    event.preventDefault();
                    form.submit();
                  }, true);
                }());
                ";
            }
        );
        $modal = $this->uiFactory->modal()->roundtrip(
            $this->lng->txt('activate_only_current'),
            [$this->uiFactory->legacy($form->getHTML()), $message]
        )->withActionButtons([$submitBtn]);
        $action_button = $this->uiFactory->button()->standard($this->lng->txt('activate_post'), '#')->withOnClick(
            $modal->getShowSignal()
        );
        $this->modalActionsContainer[] = $modal;
        $this->ctrl->clearParameters($this);

        return $action_button;
    }

    public function getCensorshipFormHTML(): string
    {
        /** @var ilForum $frm */
        $frm = $this->object->Forum;
        $form_tpl = new ilTemplate('tpl.frm_censorship_post_form.html', true, true, 'components/ILIAS/Forum');

        $form_tpl->setVariable('ANCHOR', $this->objCurrentPost->getId());
        $form_tpl->setVariable('SPACER', '<hr noshade="noshade" width="100%" size="1" align="center" />');
        $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
        $this->ctrl->setParameter(
            $this,
            'orderby',
            $this->getOrderByParam()
        );
        $form_tpl->setVariable('FORM_ACTION', $this->ctrl->getFormAction($this, 'viewThread'));
        $this->ctrl->clearParameters($this);
        $form_tpl->setVariable('TXT_CENS_MESSAGE', $this->lng->txt('forums_the_post'));
        $form_tpl->setVariable('TXT_CENS_COMMENT', $this->lng->txt('forums_censor_comment') . ':');
        $form_tpl->setVariable('CENS_MESSAGE', $frm->prepareText((string) $this->objCurrentPost->getCensorshipComment(), 2));

        if ($this->objCurrentPost->isCensored()) {
            $form_tpl->setVariable('TXT_CENS', $this->lng->txt('forums_info_censor2_post'));
            $form_tpl->setVariable('YES_BUTTON', $this->lng->txt('confirm'));
            $form_tpl->setVariable('NO_BUTTON', $this->lng->txt('cancel'));
            $form_tpl->setVariable('CMD_REVOKE_CENSORSHIP', 'revokeCensorship');
            $form_tpl->setVariable('CMD_CANCEL_REVOKE_CENSORSHIP', 'viewThread');
        } else {
            $form_tpl->setVariable('TXT_CENS', $this->lng->txt('forums_info_censor_post'));
            $form_tpl->setVariable('CANCEL_BUTTON', $this->lng->txt('cancel'));
            $form_tpl->setVariable('CONFIRM_BUTTON', $this->lng->txt('confirm'));
            $form_tpl->setVariable('CMD_ADD_CENSORSHIP', 'addCensorship');
            $form_tpl->setVariable('CMD_CANCEL_ADD_CENSORSHIP', 'viewThread');
        }

        return $form_tpl->get();
    }

    private function initReplyEditForm(): void
    {
        $isReply = in_array($this->requestAction, ['showreply', 'ready_showreply']);
        $isDraft = in_array($this->requestAction, ['publishDraft', 'editdraft']);

        $draft_id = $this->retrieveDraftId();

        // init objects
        $oForumObjects = $this->getForumObjects();
        $frm = $oForumObjects['frm'];
        $oFDForum = $oForumObjects['file_obj'];

        $this->replyEditForm = new ilPropertyFormGUI();
        $this->replyEditForm->setId('id_showreply');
        $this->replyEditForm->setTableWidth('100%');
        $cancel_cmd = 'cancelPost';
        if (in_array($this->requestAction, ['showreply', 'ready_showreply'])) {
            $this->ctrl->setParameter($this, 'action', 'ready_showreply');
        } elseif (in_array($this->requestAction, ['showdraft', 'editdraft'])) {
            $this->ctrl->setParameter($this, 'action', $this->requestAction);
            $this->ctrl->setParameter($this, 'draft_id', $draft_id);
        } else {
            $this->ctrl->setParameter($this, 'action', 'ready_showedit');
        }

        $this->ctrl->setParameter($this, 'page', (int) ($this->httpRequest->getQueryParams()['page'] ?? 0));
        $this->ctrl->setParameter(
            $this,
            'orderby',
            $this->getOrderByParam()
        );
        $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
        if ($this->isTopLevelReplyCommand()) {
            $this->replyEditForm->setFormAction(
                $this->ctrl->getFormAction($this, 'saveTopLevelPost', 'frm_page_bottom')
            );
        } elseif (in_array($this->requestAction, ['publishDraft', 'editdraft'])) {
            $this->replyEditForm->setFormAction(
                $this->ctrl->getFormAction($this, 'publishDraft', (string) $this->objCurrentPost->getId())
            );
        } else {
            $this->replyEditForm->setFormAction(
                $this->ctrl->getFormAction($this, 'savePost', (string) $this->objCurrentPost->getId())
            );
        }
        $this->ctrl->clearParameters($this);

        if ($isReply) {
            $this->replyEditForm->setTitle($this->lng->txt('forums_your_reply'));
        } elseif ($isDraft) {
            $this->replyEditForm->setTitle($this->lng->txt('forums_edit_draft'));
        } else {
            $this->replyEditForm->setTitle($this->lng->txt('forums_edit_post'));
        }

        if (
            $this->isWritingWithPseudonymAllowed() &&
            in_array($this->requestAction, ['showreply', 'ready_showreply', 'editdraft'])
        ) {
            $oAnonymousNameGUI = new ilTextInputGUI($this->lng->txt('forums_your_name'), 'alias');
            $oAnonymousNameGUI->setMaxLength(64);
            $oAnonymousNameGUI->setInfo(sprintf($this->lng->txt('forums_use_alias'), $this->lng->txt('forums_anonymous')));

            $this->replyEditForm->addItem($oAnonymousNameGUI);
        }

        $oSubjectGUI = new ilTextInputGUI($this->lng->txt('forums_subject'), 'subject');
        $oSubjectGUI->setMaxLength(255);
        $oSubjectGUI->setRequired(true);

        if ($this->objProperties->getSubjectSetting() === 'empty_subject') {
            $oSubjectGUI->setInfo($this->lng->txt('enter_new_subject'));
        }

        $this->replyEditForm->addItem($oSubjectGUI);

        $oPostGUI = new ilTextAreaInputGUI(
            $isReply ? $this->lng->txt('forums_your_reply') : $this->lng->txt('forums_edit_post'),
            'message'
        );
        $oPostGUI->setRequired(true);
        $oPostGUI->setRows(15);
        $oPostGUI->setUseRte(true);

        $quotingAllowed = (
            !$this->isTopLevelReplyCommand() && (
                ($isReply && $this->objCurrentPost->getDepth() >= 2) ||
                (!$isDraft && !$isReply && $this->objCurrentPost->getDepth() > 2) ||
                ($isDraft && $this->objCurrentPost->getDepth() >= 2)
            )
        );

        $oPostGUI->removePlugin('advlink');
        $oPostGUI->setRTERootBlockElement('');
        $oPostGUI->usePurifier(true);
        $oPostGUI->disableButtons([
            'charmap',
            'undo',
            'redo',
            'alignleft',
            'aligncenter',
            'alignright',
            'alignjustify',
            'anchor',
            'fullscreen',
            'cut',
            'copy',
            'paste',
            'pastetext',
            'formatselect'
        ]);

        if (in_array($this->requestAction, ['showreply', 'ready_showreply', 'showdraft', 'editdraft'])) {
            $oPostGUI->setRTESupport(
                $this->user->getId(),
                'frm~',
                'frm_post',
                'tpl.tinymce_frm_post.js',
                false,
                '5.6.0'
            );
        } else {
            $oPostGUI->setRTESupport(
                $this->objCurrentPost->getId(),
                'frm',
                'frm_post',
                'tpl.tinymce_frm_post.js',
                false,
                '5.6.0'
            );
        }

        $oPostGUI->setPurifier(ilHtmlPurifierFactory::getInstanceByType('frm_post'));

        $this->replyEditForm->addItem($oPostGUI);

        $umail = new ilMail($this->user->getId());
        if (!$this->user->isAnonymous() &&
            !$this->objProperties->isAnonymized() &&
            $this->rbac->system()->checkAccess('internal_mail', $umail->getMailObjectReferenceId()) &&
            !$frm->isThreadNotificationEnabled($this->user->getId(), $this->objCurrentPost->getThreadId())) {
            $oNotificationGUI = new ilCheckboxInputGUI($this->lng->txt('forum_direct_notification'), 'notify');
            $oNotificationGUI->setInfo($this->lng->txt('forum_notify_me'));
            $this->replyEditForm->addItem($oNotificationGUI);
        }

        if ($this->objProperties->isFileUploadAllowed()) {
            $oFileUploadGUI = new ilFileWizardInputGUI($this->lng->txt('forums_attachments_add'), 'userfile');
            $oFileUploadGUI->setFilenames([0 => '']);
            $this->replyEditForm->addItem($oFileUploadGUI);
        }

        $attachments_of_node = $oFDForum->getFilesOfPost();
        if (count($attachments_of_node) && in_array($this->requestAction, ['showedit', 'ready_showedit'])) {
            $oExistingAttachmentsGUI = new ilCheckboxGroupInputGUI($this->lng->txt('forums_delete_file'), 'del_file');
            foreach ($oFDForum->getFilesOfPost() as $file) {
                $oExistingAttachmentsGUI->addOption(new ilCheckboxOption($file['name'], $file['md5']));
            }
            $this->replyEditForm->addItem($oExistingAttachmentsGUI);
        }

        if (ilForumPostDraft::isAutoSavePostDraftAllowed()) {
            if (in_array($this->requestAction, ['showdraft', 'editdraft'])) {
                $draftInfoGUI = new ilNonEditableValueGUI('', 'autosave_info', true);
                $draftInfoGUI->setValue(sprintf(
                    $this->lng->txt('autosave_draft_info'),
                    ilForumPostDraft::lookupAutosaveInterval()
                ));
                $this->replyEditForm->addItem($draftInfoGUI);
            } elseif (!in_array($this->requestAction, ['showedit', 'ready_showedit'])) {
                $draftInfoGUI = new ilNonEditableValueGUI('', 'autosave_info', true);
                $draftInfoGUI->setValue(sprintf(
                    $this->lng->txt('autosave_post_draft_info'),
                    ilForumPostDraft::lookupAutosaveInterval()
                ));
                $this->replyEditForm->addItem($draftInfoGUI);
            }
        }

        $selected_draft_id = $draft_id;
        $draftObj = new ilForumPostDraft(
            $this->user->getId(),
            $this->objCurrentPost->getId(),
            $selected_draft_id
        );
        if ($draftObj->getDraftId() > 0) {
            $oFDForumDrafts = new ilFileDataForumDrafts(0, $draftObj->getDraftId());
            $files_of_draft = $oFDForumDrafts->getFilesOfPost();

            if ($files_of_draft !== []) {
                $oExistingAttachmentsGUI = new ilCheckboxGroupInputGUI(
                    $this->lng->txt('forums_delete_file'),
                    'del_file'
                );
                foreach ($files_of_draft as $file) {
                    $oExistingAttachmentsGUI->addOption(new ilCheckboxOption($file['name'], $file['md5']));
                }
                $this->replyEditForm->addItem($oExistingAttachmentsGUI);
            }
        }

        if ($this->isTopLevelReplyCommand()) {
            $this->replyEditForm->addCommandButton('saveTopLevelPost', $this->lng->txt('create'));
        } elseif ($this->requestAction === 'editdraft' && ilForumPostDraft::isSavePostDraftAllowed()) {
            $this->replyEditForm->addCommandButton('publishDraft', $this->lng->txt('publish'));
        } else {
            $this->replyEditForm->addCommandButton('savePost', $this->lng->txt('save'));
        }
        $hidden_draft_id = new ilHiddenInputGUI('draft_id');
        $auto_save_draft_id = $this->retrieveDraftId();

        $hidden_draft_id->setValue((string) $auto_save_draft_id);
        $this->replyEditForm->addItem($hidden_draft_id);

        if (in_array($this->requestAction, ['showreply', 'ready_showreply', 'editdraft'])) {
            $show_rte = $this->http->wrapper()->post()->retrieve(
                'show_rte',
                $this->refinery->byTrying([$this->refinery->kindlyTo()->int(), $this->refinery->always(0)])
            );

            if ($show_rte) {
                ilObjAdvancedEditing::_setRichTextEditorUserState($show_rte);
            }

            if ($quotingAllowed) {
                $this->replyEditForm->addCommandButton('quotePost', $this->lng->txt('forum_add_quote'));
            }

            if (!$this->user->isAnonymous() &&
                in_array($this->requestAction, ['editdraft', 'showreply', 'ready_showreply']) &&
                ilForumPostDraft::isSavePostDraftAllowed()) {
                if (ilForumPostDraft::isAutoSavePostDraftAllowed()) {
                    $this->decorateWithAutosave($this->replyEditForm);
                }

                if ($this->requestAction === 'editdraft') {
                    $this->replyEditForm->addCommandButton('updateDraft', $this->lng->txt('save_message'));
                } elseif ($this->isTopLevelReplyCommand()) {
                    $this->replyEditForm->addCommandButton('saveTopLevelDraft', $this->lng->txt('save_message'));
                } else {
                    $this->replyEditForm->addCommandButton('saveAsDraft', $this->lng->txt('save_message'));
                }

                $cancel_cmd = 'cancelDraft';
            }
        }
        $this->replyEditForm->addCommandButton($cancel_cmd, $this->lng->txt('cancel'));
    }

    private function getReplyEditForm(): ilPropertyFormGUI
    {
        if ($this->replyEditForm === null) {
            $this->initReplyEditForm();
        }

        return $this->replyEditForm;
    }

    public function createTopLevelPostObject(): void
    {
        $draft_obj = null;
        $draft_id = $this->retrieveDraftId();

        if ($draft_id > 0 && !$this->user->isAnonymous()
            && ilForumPostDraft::isSavePostDraftAllowed()) {
            $draft_obj = new ilForumPostDraft(
                $this->user->getId(),
                $this->objCurrentPost->getId(),
                $draft_id
            );
        }

        if ($draft_obj instanceof ilForumPostDraft && $draft_obj->getDraftId() > 0) {
            $this->ctrl->setParameter($this, 'action', 'editdraft');
            $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
            $this->ctrl->setParameter($this, 'draft_id', $draft_obj->getDraftId());
            $this->ctrl->setParameter($this, 'page', 0);
            $this->ctrl->setParameter(
                $this,
                'orderby',
                $this->getOrderByParam()
            );
            $this->ctrl->redirect($this, 'editDraft');
        } else {
            $this->viewThreadObject();
        }
    }

    public function saveTopLevelPostObject(): void
    {
        $this->savePostObject();
    }

    public function publishSelectedDraftObject(): void
    {
        $draft_id = $this->retrieveDraftId();
        if ($draft_id > 0) {
            $this->publishDraftObject(false);
        }
    }

    public function publishDraftObject(bool $use_replyform = true): void
    {
        if (!ilForumPostDraft::isSavePostDraftAllowed() ||
            !$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('add_reply', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if ($this->objCurrentTopic->getId() === 0) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_thr_deleted'), true);
            $this->ctrl->redirect($this);
        }

        if ($this->objCurrentTopic->isClosed()) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_thr_closed'), true);
            $this->ctrl->redirect($this);
        }

        if ($this->objCurrentPost->getId() === 0) {
            $this->requestAction = '';
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_parent_deleted'));
            $this->viewThreadObject();
            return;
        }

        $draft = new ilForumPostDraft($this->user->getId(), $this->objCurrentPost->getId(), $this->retrieveDraftId());
        $this->checkDraftAccess($draft->getDraftId());

        if ($use_replyform) {
            $oReplyEditForm = $this->getReplyEditForm();
            if (!$oReplyEditForm->checkInput()) {
                $oReplyEditForm->setValuesByPost();
                $this->viewThreadObject();
                return;
            }
            $post_subject = $oReplyEditForm->getInput('subject');
            $post_message = $oReplyEditForm->getInput('message');
            $mob_direction = 0;
        } else {
            $post_subject = $draft->getPostSubject();
            $post_message = $draft->getPostMessage();
            $mob_direction = 1;
        }

        $oForumObjects = $this->getForumObjects();
        $frm = $oForumObjects['frm'];
        $frm->setMDB2WhereCondition(' top_frm_fk = %s ', ['integer'], [$frm->getForumId()]);

        // reply: new post
        $status = true;
        $send_activation_mail = false;

        if ($this->objProperties->isPostActivationEnabled()) {
            if (!$this->is_moderator) {
                $status = false;
                $send_activation_mail = true;
            } elseif ($this->objCurrentPost->isAnyParentDeactivated()) {
                $status = false;
            }
        }

        $newPost = $frm->generatePost(
            $draft->getForumId(),
            $draft->getThreadId(),
            $this->user->getId(),
            $draft->getPostDisplayUserId(),
            ilRTE::_replaceMediaObjectImageSrc($post_message, $mob_direction),
            $draft->getPostId(),
            $draft->isNotificationEnabled(),
            $this->handleFormInput($post_subject, false),
            $draft->getPostUserAlias(),
            '',
            $status,
            $send_activation_mail
        );

        $this->object->markPostRead(
            $this->user->getId(),
            $this->objCurrentTopic->getId(),
            $this->objCurrentPost->getId()
        );

        $uploadedObjects = ilObjMediaObject::_getMobsOfObject('frm~:html', $this->user->getId());

        foreach ($uploadedObjects as $mob) {
            ilObjMediaObject::_removeUsage($mob, 'frm~:html', $this->user->getId());
            ilObjMediaObject::_saveUsage($mob, 'frm:html', $newPost);
        }
        ilForumUtil::saveMediaObjects($post_message, 'frm:html', $newPost, $mob_direction);
        $post_obj = new ilForumPost($newPost);

        if ($draft->getRCID() > 0) {
            $post_obj->setRCID($draft->getRCID());
            $post_obj->update();
        }

        if ($this->objProperties->isFileUploadAllowed()) {
            $file = $_FILES['userfile'] ?? [];
            if (is_array($file) && !empty($file)) {
                $tmp_file_obj = new ilFileDataForum($this->object->getId(), $newPost);
                $tmp_file_obj->storeUploadedFiles();
            }
        }

        $GLOBALS['ilAppEventHandler']->raise(
            'components/ILIAS/Forum',
            'publishedDraft',
            [
                'draftObj' => $draft,
                'obj_id' => $this->object->getId(),
                'is_file_upload_allowed' => $this->objProperties->isFileUploadAllowed()
            ]
        );
        $draft->deleteDraft();

        $GLOBALS['ilAppEventHandler']->raise(
            'components/ILIAS/Forum',
            'createdPost',
            [
                'object' => $this->object,
                'ref_id' => $this->object->getRefId(),
                'post' => $post_obj,
                'notify_moderators' => $send_activation_mail
            ]
        );

        $message = '';
        if (!$this->is_moderator && !$status) {
            $message .= $this->lng->txt('forums_post_needs_to_be_activated');
        } else {
            $message .= $this->lng->txt('forums_post_new_entry');
        }

        $frm_session_values = ilSession::get('frm');
        if (is_array($frm_session_values)) {
            $frm_session_values[$this->objCurrentTopic->getId()]['openTreeNodes'][] = $this->objCurrentPost->getId();
        }
        ilSession::set('frm', $frm_session_values);

        $this->ctrl->clearParameters($this);
        $this->tpl->setOnScreenMessage('success', $message, true);
        $this->ctrl->setParameter($this, 'pos_pk', $newPost);
        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());

        $this->ctrl->redirect($this, 'viewThread');
    }

    public function savePostObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if ($this->objCurrentTopic->getId() === 0) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_thr_deleted'), true);
            $this->ctrl->redirect($this);
        }

        if ($this->objCurrentTopic->isClosed()) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_thr_closed'), true);
            $this->ctrl->redirect($this);
        }

        $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentTopic);

        $oReplyEditForm = $this->getReplyEditForm();
        if ($oReplyEditForm->checkInput()) {
            if ($this->objCurrentPost->getId() === 0) {
                $this->requestAction = '';
                $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_parent_deleted'), true);
                $this->viewThreadObject();
                return;
            }

            $oForumObjects = $this->getForumObjects();
            $forumObj = $oForumObjects['forumObj'];
            $frm = $oForumObjects['frm'];
            $frm->setMDB2WhereCondition(' top_frm_fk = %s ', ['integer'], [$frm->getForumId()]);
            $topicData = $frm->getOneTopic();

            $autosave_draft_id = $this->http->wrapper()->post()->retrieve(
                'draft_id',
                $this->refinery->byTrying([$this->refinery->kindlyTo()->int(), $this->refinery->always(0)])
            );

            if ($this->requestAction === 'ready_showreply') {
                if (!$this->access->checkAccess('add_reply', '', $this->object->getRefId())) {
                    $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
                }

                $status = true;
                $send_activation_mail = false;

                if ($this->objProperties->isPostActivationEnabled()) {
                    if (!$this->is_moderator) {
                        $status = false;
                        $send_activation_mail = true;
                    } elseif ($this->objCurrentPost->isAnyParentDeactivated()) {
                        $status = false;
                    }
                }

                if ($this->isWritingWithPseudonymAllowed()) {
                    if ((string) $oReplyEditForm->getInput('alias') === '') {
                        $user_alias = $this->lng->txt('forums_anonymous');
                    } else {
                        $user_alias = $oReplyEditForm->getInput('alias');
                    }
                    $display_user_id = 0;
                } else {
                    $user_alias = $this->user->getLogin();
                    $display_user_id = $this->user->getId();
                }

                $newPost = $frm->generatePost(
                    $topicData->getTopPk(),
                    $this->objCurrentTopic->getId(),
                    $this->user->getId(),
                    $display_user_id,
                    ilRTE::_replaceMediaObjectImageSrc($oReplyEditForm->getInput('message')),
                    $this->objCurrentPost->getId(),
                    $oReplyEditForm->getInput('notify') && !$this->user->isAnonymous(),
                    $this->handleFormInput($oReplyEditForm->getInput('subject'), false),
                    $user_alias,
                    '',
                    $status,
                    $send_activation_mail
                );

                if ($autosave_draft_id > 0) {
                    $draft = new ilForumPostDraft(
                        $this->user->getId(),
                        $this->objCurrentPost->getId(),
                        $autosave_draft_id
                    );
                    if ($this->hasDraftAccess($draft)) {
                        $draft->deleteDraft();
                    }
                }

                // mantis #8115: Mark parent as read
                $this->object->markPostRead(
                    $this->user->getId(),
                    $this->objCurrentTopic->getId(),
                    $this->objCurrentPost->getId()
                );

                // copy temporary media objects (frm~)
                ilForumUtil::moveMediaObjects(
                    $oReplyEditForm->getInput('message'),
                    'frm~:html',
                    $this->user->getId(),
                    'frm:html',
                    $newPost
                );

                if ($this->objProperties->isFileUploadAllowed()) {
                    $oFDForum = new ilFileDataForum($forumObj->getId(), $newPost);
                    $file = $_FILES['userfile'];
                    if (is_array($file) && !empty($file)) {
                        $oFDForum->storeUploadedFiles();
                    }
                }

                $GLOBALS['ilAppEventHandler']->raise(
                    'components/ILIAS/Forum',
                    'createdPost',
                    [
                        'object' => $this->object,
                        'ref_id' => $this->object->getRefId(),
                        'post' => new ilForumPost($newPost),
                        'notify_moderators' => $send_activation_mail
                    ]
                );

                $message = '';
                if (!$this->is_moderator && !$status) {
                    $message .= $this->lng->txt('forums_post_needs_to_be_activated');
                } else {
                    $message .= $this->lng->txt('forums_post_new_entry');
                }

                $this->tpl->setOnScreenMessage('success', $message, true);
                $this->ctrl->clearParameters($this);
                $this->ctrl->setParameter($this, 'post_created_below', $this->objCurrentPost->getId());
                $this->ctrl->setParameter($this, 'pos_pk', $newPost);
                $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
            } else {
                if ((!$this->is_moderator &&
                        !$this->objCurrentPost->isOwner($this->user->getId())) || $this->objCurrentPost->isCensored() ||
                    $this->user->isAnonymous()) {
                    $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
                }

                $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentPost->getThread());

                $oldMediaObjects = ilObjMediaObject::_getMobsOfObject('frm:html', $this->objCurrentPost->getId());
                $curMediaObjects = ilRTE::_getMediaObjects($oReplyEditForm->getInput('message'));
                foreach ($oldMediaObjects as $oldMob) {
                    $found = false;
                    foreach ($curMediaObjects as $curMob) {
                        if ($oldMob === $curMob) {
                            $found = true;
                            break;
                        }
                    }
                    if (!$found && ilObjMediaObject::_exists($oldMob)) {
                        ilObjMediaObject::_removeUsage($oldMob, 'frm:html', $this->objCurrentPost->getId());
                        $mob_obj = new ilObjMediaObject($oldMob);
                        $mob_obj->delete();
                    }
                }

                // save old activation status for send_notification decision
                $old_status_was_active = $this->objCurrentPost->isActivated();

                // if active post has been edited posting mus be activated again by moderator
                $status = true;
                $send_activation_mail = false;

                if ($this->objProperties->isPostActivationEnabled()) {
                    if (!$this->is_moderator) {
                        $status = false;
                        $send_activation_mail = true;
                    } elseif ($this->objCurrentPost->isAnyParentDeactivated()) {
                        $status = false;
                    }
                }
                $this->objCurrentPost->setStatus($status);

                $this->objCurrentPost->setSubject($this->handleFormInput($oReplyEditForm->getInput('subject'), false));
                $this->objCurrentPost->setMessage(ilRTE::_replaceMediaObjectImageSrc(
                    $oReplyEditForm->getInput('message')
                ));
                $this->objCurrentPost->setNotification($oReplyEditForm->getInput('notify') && !$this->user->isAnonymous());
                $this->objCurrentPost->setChangeDate(date('Y-m-d H:i:s'));
                $this->objCurrentPost->setUpdateUserId($this->user->getId());

                if ($this->objCurrentPost->update()) {
                    $this->objCurrentPost->reload();

                    // Change news item accordingly
                    // note: $this->objCurrentPost->getForumId() does not give us the forum ID here (why?)
                    $news_id = ilNewsItem::getFirstNewsIdForContext(
                        $forumObj->getId(),
                        'frm',
                        $this->objCurrentPost->getId(),
                        'pos'
                    );
                    if ($news_id > 0) {
                        $news_item = new ilNewsItem($news_id);
                        $news_item->setTitle($this->objCurrentPost->getSubject());
                        $news_item->setContent(
                            ilRTE::_replaceMediaObjectImageSrc($frm->prepareText(
                                $this->objCurrentPost->getMessage()
                            ), 1)
                        );

                        if ($this->objCurrentPost->getMessage() !== strip_tags($this->objCurrentPost->getMessage())) {
                            $news_item->setContentHtml(true);
                        } else {
                            $news_item->setContentHtml(false);
                        }
                        $news_item->update();
                    }

                    $oFDForum = $oForumObjects['file_obj'];

                    $file2delete = $oReplyEditForm->getInput('del_file');
                    if (is_array($file2delete) && count($file2delete)) {
                        $oFDForum->unlinkFilesByMD5Filenames($file2delete);
                    }

                    if ($this->objProperties->isFileUploadAllowed()) {
                        $file = $_FILES['userfile'];
                        if (is_array($file) && !empty($file)) {
                            $oFDForum->storeUploadedFiles();
                        }
                    }

                    $GLOBALS['ilAppEventHandler']->raise(
                        'components/ILIAS/Forum',
                        'updatedPost',
                        [
                            'ref_id' => $this->object->getRefId(),
                            'post' => $this->objCurrentPost,
                            'notify_moderators' => $send_activation_mail,
                            'old_status_was_active' => $old_status_was_active
                        ]
                    );

                    $this->tpl->setOnScreenMessage('success', $this->lng->txt('forums_post_modified'), true);
                }

                $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
                $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
                $this->ctrl->setParameter($this, 'viewmode', $this->selectedSorting);
            }
            $this->ctrl->redirect($this, 'viewThread');
        } else {
            $this->requestAction = substr($this->requestAction, 6);
        }
        $this->viewThreadObject();
    }

    private function hideToolbar($a_flag = null)
    {
        if ($a_flag === null) {
            return $this->hideToolbar;
        }

        $this->hideToolbar = $a_flag;
        return $this;
    }

    public function quotePostObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if ($this->objCurrentTopic->isClosed()) {
            $this->requestAction = '';
            $this->viewThreadObject();
            return;
        }

        $oReplyEditForm = $this->getReplyEditForm();

        $oReplyEditForm->getItemByPostVar('subject')->setRequired(false);
        $oReplyEditForm->getItemByPostVar('message')->setRequired(false);

        $oReplyEditForm->checkInput();

        $oReplyEditForm->getItemByPostVar('subject')->setRequired(true);
        $oReplyEditForm->getItemByPostVar('message')->setRequired(true);

        $this->requestAction = 'showreply';

        $this->viewThreadObject();
    }

    /**
     * @return array{forumObj: ilObjForum, frm: ilForum, file_obj: ilFileDataForum}
     */
    private function getForumObjects(): array
    {
        if ($this->forumObjects === null) {
            $forumObj = $this->object;
            $file_obj = new ilFileDataForum($forumObj->getId(), $this->objCurrentPost->getId());
            $frm = $forumObj->Forum;
            $frm->setForumId($forumObj->getId());
            $frm->setForumRefId($forumObj->getRefId());

            $this->forumObjects['forumObj'] = $forumObj;
            $this->forumObjects['frm'] = $frm;
            $this->forumObjects['file_obj'] = $file_obj;
        }

        return $this->forumObjects;
    }

    public function checkUsersViewMode(): void
    {
        $this->selectedSorting = $this->objProperties->getDefaultView();

        if ($this->getObject() !== null) {
            $view_mode = 'viewmode_' . $this->getObject()->getId();
            if (in_array((int) ilSession::get($view_mode), [
                ilForumProperties::VIEW_TREE,
                ilForumProperties::VIEW_DATE_ASC,
                ilForumProperties::VIEW_DATE_DESC
            ], true)) {
                $this->selectedSorting = ilSession::get($view_mode);
            }

            if (
                isset($this->httpRequest->getQueryParams()['viewmode']) &&
                (int) $this->httpRequest->getQueryParams()['viewmode'] !== $this->selectedSorting
            ) {
                $this->selectedSorting = (int) $this->httpRequest->getQueryParams()['viewmode'];
            }

            if (!in_array($this->selectedSorting, [
                ilForumProperties::VIEW_TREE,
                ilForumProperties::VIEW_DATE_ASC,
                ilForumProperties::VIEW_DATE_DESC
            ], true)) {
                $this->selectedSorting = $this->objProperties->getDefaultView();
            }

            ilSession::set($view_mode, $this->selectedSorting);
        }
    }

    public function resetLimitedViewObject(): void
    {
        $this->selected_post_storage->set($this->objCurrentTopic->getId(), 0);
        $this->ctrl->redirect($this, 'viewThread');
    }

    public function viewThreadObject(): void
    {
        $thr_pk = $this->objCurrentTopic->getId();

        $bottom_toolbar = clone $this->toolbar;
        $toolbar_items = [];

        // quick and dirty: check for treeview
        $thread_control_session_values = ilSession::get('thread_control');
        if (is_array($thread_control_session_values)) {
            if (!isset($thread_control_session_values['old'])) {
                $thread_control_session_values['old'] = $thr_pk;
                $thread_control_session_values['new'] = $thr_pk;
                ilSession::set('thread_control', $thread_control_session_values);
            } elseif (isset($thread_control_session_values['old']) && $thr_pk !== $thread_control_session_values['old']) {
                $thread_control_session_values['new'] = $thr_pk;
                ilSession::set('thread_control', $thread_control_session_values);
            }
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $oForumObjects = $this->getForumObjects();
        $forumObj = $oForumObjects['forumObj'];
        $frm = $oForumObjects['frm'];
        $file_obj = $oForumObjects['file_obj'];

        $selected_draft_id = (int) ($this->httpRequest->getQueryParams()['draft_id'] ?? 0);
        if (isset($this->httpRequest->getQueryParams()['file'])) {
            $file_obj_for_delivery = $file_obj;
            if ($selected_draft_id > 0 && ilForumPostDraft::isSavePostDraftAllowed()) {
                $file_obj_for_delivery = new ilFileDataForumDrafts($forumObj->getId(), $selected_draft_id);
            }
            $file_obj_for_delivery->deliverFile(ilUtil::stripSlashes($this->httpRequest->getQueryParams()['file']));
        }

        if ($this->objCurrentTopic->getId() === 0) {
            $this->ctrl->redirect($this, 'showThreads');
        }

        $pageIndex = 0;
        if (isset($this->httpRequest->getQueryParams()['page'])) {
            $pageIndex = max((int) $this->httpRequest->getQueryParams()['page'], $pageIndex);
        }

        if ($this->selected_post_storage->get($this->objCurrentTopic->getId()) > 0) {
            $firstNodeInThread = new ilForumPost(
                $this->selected_post_storage->get($this->objCurrentTopic->getId()),
                $this->is_moderator,
                false
            );
        } else {
            $firstNodeInThread = $this->objCurrentTopic->getPostRootNode();
        }

        $toolContext = $this->globalScreen
            ->tool()
            ->context()
            ->current();

        $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentTopic);

        $threadContentTemplate = new ilTemplate(
            'tpl.forums_threads_view.html',
            true,
            true,
            'components/ILIAS/Forum'
        );


        if ($this->selectedSorting === ilForumProperties::VIEW_TREE) {
            $order_field = 'frm_posts_tree.rgt';
            $this->objCurrentTopic->setOrderDirection('DESC');
            $threadContentTemplate->setVariable('LIST_TYPE', $this->viewModeOptions[$this->selectedSorting]);
        } else {
            $order_field = 'frm_posts.pos_date';
            $this->objCurrentTopic->setOrderDirection(
                $this->selectedSorting === ilForumProperties::VIEW_DATE_DESC ? 'DESC' : 'ASC'
            );
            $threadContentTemplate->setVariable('LIST_TYPE', $this->sortationOptions[$this->selectedSorting]);
        }
        $this->objCurrentTopic->setOrderField($order_field);

        $additionalDataExists = $toolContext->getAdditionalData()->exists(ForumGlobalScreenToolsProvider::SHOW_FORUM_THREADS_TOOL);
        $subtree_nodes = $this->objCurrentTopic->getPostTree($firstNodeInThread);
        $numberOfPostings = count($subtree_nodes);
        if ($numberOfPostings > 0 && !$additionalDataExists && $this->selectedSorting === ilForumProperties::VIEW_TREE) {
            $toolContext
                ->addAdditionalData(ForumGlobalScreenToolsProvider::SHOW_FORUM_THREADS_TOOL, true)
                ->addAdditionalData(ForumGlobalScreenToolsProvider::REF_ID, $this->ref_id)
                ->addAdditionalData(ForumGlobalScreenToolsProvider::FORUM_THEAD, $this->objCurrentTopic)
                ->addAdditionalData(ForumGlobalScreenToolsProvider::FORUM_THREAD_ROOT, $firstNodeInThread)
                ->addAdditionalData(ForumGlobalScreenToolsProvider::FORUM_BASE_CONTROLLER, $this)
                ->addAdditionalData(ForumGlobalScreenToolsProvider::PAGE, $pageIndex);
        }
        // Set context for login
        $append = '_' . $this->objCurrentTopic->getId() .
            ($this->objCurrentPost->getId() !== 0 ? '_' . $this->objCurrentPost->getId() : '');
        $this->tpl->setLoginTargetPar('frm_' . $this->object->getRefId() . $append);

        // delete temporary media object (not in case a user adds media objects and wants to save an invalid form)
        if (!in_array($this->requestAction, ['showreply', 'showedit'])) {
            try {
                $mobs = ilObjMediaObject::_getMobsOfObject('frm~:html', $this->user->getId());
                foreach ($mobs as $mob) {
                    if (ilObjMediaObject::_exists($mob)) {
                        ilObjMediaObject::_removeUsage($mob, 'frm~:html', $this->user->getId());
                        $mob_obj = new ilObjMediaObject($mob);
                        $mob_obj->delete();
                    }
                }
            } catch (Exception) {
            }
        }

        if (!$this->getCreationMode() && $this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->ilNavigationHistory->addItem(
                $this->object->getRefId(),
                ilLink::_getLink($this->object->getRefId(), 'frm'),
                'frm'
            );
        }

        $this->prepareThreadScreen($forumObj);


        if (isset($this->httpRequest->getQueryParams()['anchor'])) {
            $threadContentTemplate->setVariable('JUMP2ANCHOR_ID', (int) $this->httpRequest->getQueryParams()['anchor']);
        }
        $frm->setMDB2WhereCondition('top_frm_fk = %s ', ['integer'], [$frm->getForumId()]);

        ilChangeEvent::_recordReadEvent(
            $this->object->getType(),
            $this->object->getRefId(),
            $this->object->getId(),
            $this->user->getId()
        );

        if ($firstNodeInThread) {
            $this->objCurrentTopic->updateVisits();

            $this->tpl->setTitle($this->lng->txt('forums_thread') . ' "' . $this->objCurrentTopic->getSubject() . '"');

            $this->locator->addRepositoryItems();
            $this->locator->addItem($this->object->getTitle(), $this->ctrl->getLinkTarget($this, ''), '_top');
            $this->tpl->setLocator();

            if (!$this->user->isAnonymous() &&
                $forumObj->getCountUnread($this->user->getId(), $this->objCurrentTopic->getId(), true)) {
                $this->ctrl->setParameter($this, 'mark_read', '1');
                $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());

                $mark_thr_read_button = $this->uiFactory->button()->standard(
                    $this->lng->txt('forums_mark_read'),
                    $this->ctrl->getLinkTarget($this, 'viewThread')
                );

                $toolbar_items[] = $mark_thr_read_button;

                $this->ctrl->clearParameters($this);
            }

            $this->ctrl->setParameterByClass(ilForumExportGUI::class, 'print_thread', $this->objCurrentTopic->getId());
            $this->ctrl->setParameterByClass(
                ilForumExportGUI::class,
                'thr_top_fk',
                $this->objCurrentTopic->getForumId()
            );

            if ($numberOfPostings > 0 && $this->settings->get('forum_enable_print', '0')) {
                $print_thr_button = $this->uiFactory->button()->standard(
                    $this->lng->txt('forums_print_thread'),
                    $this->ctrl->getLinkTargetByClass(ilForumExportGUI::class, 'printThread')
                );
                $toolbar_items[] = $print_thr_button;
            }


            $this->ctrl->clearParametersByClass(ilForumExportGUI::class);

            $this->addHeaderAction();

            if (isset($this->httpRequest->getQueryParams()['mark_read'])) {
                $forumObj->markThreadRead($this->user->getId(), $this->objCurrentTopic->getId());
                $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_thread_marked'), true);
            }

            if ($firstNodeInThread instanceof ilForumPost &&
                !$this->isTopLevelReplyCommand() &&
                !$this->objCurrentTopic->isClosed() &&
                $this->access->checkAccess('add_reply', '', $this->object->getRefId())) {
                $this->ctrl->setParameter($this, 'action', 'showreply');
                $this->ctrl->setParameter($this, 'pos_pk', $firstNodeInThread->getId());
                $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
                $this->ctrl->setParameter(
                    $this,
                    'page',
                    (int) ($this->httpRequest->getQueryParams()['page'] ?? 0)
                );
                $this->ctrl->setParameter(
                    $this,
                    'orderby',
                    $this->getOrderByParam()
                );
                if ($numberOfPostings > 0) {
                    $reply_button = $this->uiFactory->button()->standard(
                        $this->lng->txt('add_new_answer'),
                        $this->ctrl->getLinkTarget($this, 'createTopLevelPost', 'frm_page_bottom')
                    );
                } else {
                    $reply_button = $this->uiFactory->button()->primary(
                        $this->lng->txt('add_new_answer'),
                        $this->ctrl->getLinkTarget($this, 'createTopLevelPost', 'frm_page_bottom')
                    );
                }
                $this->ctrl->clearParameters($this);
                array_unshift($toolbar_items, $reply_button);
            }

            // no posts
            if ($numberOfPostings === 0 && $firstNodeInThread->getId() === 0) {
                $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_no_posts_available'));
            }

            $pageSize = $frm->getPageHits();
            $postIndex = 0;
            if ($numberOfPostings > $pageSize) {
                $this->ctrl->setParameter($this, 'ref_id', $this->object->getRefId());
                $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
                $this->ctrl->setParameter(
                    $this,
                    'orderby',
                    $this->getOrderByParam()
                );
                $paginationUrl = $this->ctrl->getLinkTarget($this, 'viewThread', '');
                $this->ctrl->clearParameters($this);

                $pagination = $this->uiFactory->viewControl()
                                              ->pagination()
                                              ->withTargetURL($paginationUrl, 'page')
                                              ->withTotalEntries($numberOfPostings)
                                              ->withPageSize($pageSize)
                                              ->withMaxPaginationButtons(10)
                                              ->withCurrentPage($pageIndex);

                $threadContentTemplate->setVariable('THREAD_MENU', $this->uiRenderer->render(
                    $pagination
                ));
                $threadContentTemplate->setVariable('THREAD_MENU_BOTTOM', $this->uiRenderer->render(
                    $pagination
                ));
            }

            $doRenderDrafts = ilForumPostDraft::isSavePostDraftAllowed() && !$this->user->isAnonymous();
            $draftsObjects = [];
            if ($doRenderDrafts) {
                $draftsObjects = ilForumPostDraft::getSortedDrafts(
                    $this->user->getId(),
                    $this->objCurrentTopic->getId(),
                    $this->selectedSorting
                );
            }

            $pagedPostings = array_slice($subtree_nodes, $pageIndex * $pageSize, $pageSize);

            $this->ensureValidPageForCurrentPosting($subtree_nodes, $pagedPostings, $pageSize, $firstNodeInThread);

            if ($doRenderDrafts && $pageIndex === 0 &&
                $this->selectedSorting === ilForumProperties::VIEW_DATE_DESC) {
                foreach ($draftsObjects as $draft) {
                    $referencePosting = array_values(array_filter(
                        $subtree_nodes,
                        static function (ilForumPost $post) use ($draft): bool {
                            return $draft->getPostId() === $post->getId();
                        }
                    ))[0] ?? $firstNodeInThread;

                    $this->renderDraftContent(
                        $threadContentTemplate,
                        $this->requestAction,
                        $referencePosting,
                        [$draft]
                    );
                }
            }

            foreach ($pagedPostings as $node) {
                $this->ctrl->clearParameters($this);

                if (!$this->isTopLevelReplyCommand() && $this->objCurrentPost->getId() === $node->getId() &&
                    ($this->is_moderator || $node->isActivated() || $node->isOwner($this->user->getId()))) {
                    if (!$this->objCurrentTopic->isClosed() && in_array($this->requestAction, [
                            'showreply',
                            'showedit',
                        ])) {
                        $this->renderPostingForm($threadContentTemplate, $frm, $node, $this->requestAction);
                    } elseif ($this->requestAction === 'censor' &&
                        !$this->objCurrentTopic->isClosed() && $this->is_moderator) {
                        $threadContentTemplate->setVariable('FORM', $this->getCensorshipFormHTML());
                    }
                }

                $this->renderPostContent($threadContentTemplate, $node, $this->requestAction, $pageIndex, $postIndex);
                if ($doRenderDrafts && $this->selectedSorting === ilForumProperties::VIEW_TREE) {
                    $this->renderDraftContent(
                        $threadContentTemplate,
                        $this->requestAction,
                        $node,
                        $draftsObjects[$node->getId()] ?? []
                    );
                }

                $postIndex++;
            }

            if (
                $this->selectedSorting === ilForumProperties::VIEW_DATE_ASC &&
                $doRenderDrafts && $pageIndex === max(0, (int) (ceil($numberOfPostings / $pageSize) - 1))
            ) {
                foreach ($draftsObjects as $draft) {
                    $referencePosting = array_values(array_filter(
                        $subtree_nodes,
                        static function (ilForumPost $post) use ($draft): bool {
                            return $draft->getPostId() === $post->getId();
                        }
                    ))[0] ?? $firstNodeInThread;

                    $this->renderDraftContent(
                        $threadContentTemplate,
                        $this->requestAction,
                        $referencePosting,
                        [$draft]
                    );
                }
            }

            if (
                $firstNodeInThread instanceof ilForumPost && $doRenderDrafts &&
                $this->selectedSorting === ilForumProperties::VIEW_TREE
            ) {
                $this->renderDraftContent(
                    $threadContentTemplate,
                    $this->requestAction,
                    $firstNodeInThread,
                    $draftsObjects[$firstNodeInThread->getId()] ?? []
                );
            }

            if ($firstNodeInThread instanceof ilForumPost &&
                !$this->objCurrentTopic->isClosed() &&
                in_array($this->ctrl->getCmd(), ['createTopLevelPost', 'saveTopLevelPost', 'saveTopLevelDraft'], true) &&
                $this->access->checkAccess('add_reply', '', $this->object->getRefId())) {
                // Important: Don't separate the following two lines (very fragile code ...)
                $this->objCurrentPost->setId($firstNodeInThread->getId());
                $form = $this->getReplyEditForm();

                if (in_array($this->ctrl->getCmd(), ['saveTopLevelPost', 'saveTopLevelDraft'])) {
                    $form->setValuesByPost();
                }
                $threadContentTemplate->setVariable('BOTTOM_FORM', $form->getHTML());
            }
        } else {
            $threadContentTemplate->setCurrentBlock('posts_no');
            $threadContentTemplate->setVariable(
                'TXT_MSG_NO_POSTS_AVAILABLE',
                $this->lng->txt('forums_posts_not_available')
            );
            $threadContentTemplate->parseCurrentBlock();
        }

        $to_top_button = $this->uiFactory->button()
                                         ->standard($this->lng->txt('top_of_page'), '#frm_page_top');
        $bottom_toolbar->addComponent($to_top_button);
        if ($numberOfPostings > 0) {
            $threadContentTemplate->setVariable('TOOLBAR_BOTTOM', $bottom_toolbar->getHTML());
        }

        if ($toolbar_items !== []) {
            foreach ($toolbar_items as $component_index => $item) {
                $this->toolbar->addComponent($item);
                $bottom_toolbar->addComponent($item);
                if ($component_index === 0 && $numberOfPostings > 0) {
                    $this->renderViewModeControl($this->selectedSorting);
                    if ($this->selectedSorting !== ilForumProperties::VIEW_TREE) {
                        $this->renderSortationControl($this->selectedSorting);
                    }
                }
            }
        }

        $this->tpl->setPermanentLink(
            $this->object->getType(),
            $this->object->getRefId(),
            (string) $this->objCurrentTopic->getId()
        );

        $this->tpl->addOnLoadCode(
            <<<EOD
                document.querySelectorAll('.ilFrmPostContent img').forEach((img) => {
                  const maxWidth = img.getAttribute('width');
                  const maxHeight = img.getAttribute('height');
                
                  if (maxWidth) {
                    img.style.maxWidth = maxWidth + 'px';
                    img.removeAttribute('width');
                  }
                
                  if (maxHeight) {
                    img.style.maxHeight = maxHeight + 'px';
                    img.removeAttribute('height');
                  }
                });
EOD
        );

        if ($this->selectedSorting === ilForumProperties::VIEW_TREE && ($this->selected_post_storage->get($thr_pk) > 0)) {
            $info = $this->getResetLimitedViewInfo();
        }

        $this->tpl->setContent(($info ?? '') . $threadContentTemplate->get() . $this->getModalActions());
    }

    private function renderViewModeControl(int $currentViewMode): void
    {
        if ($currentViewMode === 3) {
            $currentViewMode = 2;
        }
        $translationKeys = [];
        foreach ($this->viewModeOptions as $sortingConstantKey => $languageKey) {
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
            $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
            $this->ctrl->setParameter($this, 'viewmode', $sortingConstantKey);

            $translationKeys[$this->lng->txt($languageKey)] = $this->ctrl->getLinkTarget(
                $this,
                'viewThread',
                ''
            );

            $this->ctrl->clearParameters($this);
        }

        if ($currentViewMode > ilForumProperties::VIEW_DATE_ASC) {
            $currentViewMode = ilForumProperties::VIEW_DATE_ASC;
        }

        $sortViewControl = $this->uiFactory
            ->viewControl()
            ->mode($translationKeys, $this->lng->txt($this->viewModeOptions[$currentViewMode]))
            ->withActive($this->lng->txt($this->viewModeOptions[$currentViewMode]));
        $this->toolbar->addComponent($sortViewControl);
    }

    private function renderSortationControl(int $currentSorting): void
    {
        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
        $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
        $target = $this->ctrl->getLinkTarget(
            $this,
            'viewThread',
            ''
        );

        $translatedSortationOptions = array_map(function ($value): string {
            return $this->lng->txt($value);
        }, $this->sortationOptions);

        $sortingDirectionViewControl = $this->uiFactory
            ->viewControl()
            ->sortation($translatedSortationOptions, (string) $currentSorting)
            ->withTargetURL($target, 'viewmode');
        $this->toolbar->addComponent($sortingDirectionViewControl);
    }

    private function getModifiedReOnSubject(): string
    {
        return (new \ILIAS\components\Forum\Subject\PostingReplySubjectBuilder(
            $this->lng->txt('post_reply'),
            $this->lng->txt('post_reply_count')
        ))->build($this->objCurrentPost->getSubject());
    }

    public function showUserObject(): void
    {
        $user_id = $this->http->wrapper()->query()->retrieve(
            'user',
            $this->refinery->byTrying([$this->refinery->kindlyTo()->int(), $this->refinery->always(0)])
        );
        $backurl = $this->http->wrapper()->query()->retrieve(
            'backurl',
            $this->refinery->byTrying([$this->refinery->kindlyTo()->string(), $this->refinery->always('')])
        );

        $profile_gui = new ilPublicUserProfileGUI($user_id);
        $add = $this->getUserProfileAdditional($this->object->getRefId(), $user_id);
        $profile_gui->setAdditional($add);
        $profile_gui->setBackUrl(ilUtil::stripSlashes($backurl));
        $this->tpl->setContent($this->ctrl->getHTML($profile_gui));
    }

    protected function getUserProfileAdditional(int $a_forum_ref_id, int $a_user_id): array
    {
        if (!$this->access->checkAccess('read', '', $a_forum_ref_id)) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        /** @var ilObjForum $ref_obj */
        $ref_obj = ilObjectFactory::getInstanceByRefId($a_forum_ref_id);
        if ($ref_obj->getType() === 'frm') {
            $forumObj = new ilObjForum($a_forum_ref_id);
            $frm = $forumObj->Forum;
            $frm->setForumId($forumObj->getId());
            $frm->setForumRefId($forumObj->getRefId());
        } else {
            $frm = new ilForum();
        }

        if ($this->access->checkAccess('moderate_frm', '', $a_forum_ref_id)) {
            $numPosts = $frm->countUserArticles($a_user_id);
        } else {
            $numPosts = $frm->countActiveUserArticles($a_user_id);
        }

        return [$this->lng->txt('forums_posts') => $numPosts];
    }

    public function performThreadsActionObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        ilSession::set('threads2move', []);

        $thread_ids = $this->retrieveThreadIds();
        $cmd = $this->ctrl->getCmd();
        $message = null;

        if ($thread_ids !== []) {
            if ($cmd === 'move') {
                if ($this->is_moderator) {
                    ilSession::set('threads2move', $thread_ids);
                    $this->moveThreadsObject();
                }
            } elseif ($cmd === 'enable_notifications' && (int) $this->settings->get('forum_notification', '0') !== 0) {
                foreach ($thread_ids as $thread_id) {
                    $tmp_obj = new ilForumTopic($thread_id);
                    $this->ensureThreadBelongsToForum($this->object->getId(), $tmp_obj);
                    $tmp_obj->enableNotification($this->user->getId());
                }

                $this->ctrl->redirect($this, 'showThreads');
            } elseif ($cmd === 'disable_notifications' && (int) $this->settings->get('forum_notification', '0') !== 0) {
                foreach ($thread_ids as $thread_id) {
                    $tmp_obj = new ilForumTopic($thread_id);
                    $this->ensureThreadBelongsToForum($this->object->getId(), $tmp_obj);
                    $tmp_obj->disableNotification($this->user->getId());
                }

                $this->ctrl->redirect($this, 'showThreads');
            } elseif ($cmd === 'close') {
                if ($this->is_moderator) {
                    foreach ($thread_ids as $thread_id) {
                        $tmp_obj = new ilForumTopic($thread_id);
                        $this->ensureThreadBelongsToForum($this->object->getId(), $tmp_obj);
                        $tmp_obj->close();
                    }
                }
                $this->tpl->setOnScreenMessage('success', $this->lng->txt('selected_threads_closed'), true);
                $this->ctrl->redirect($this, 'showThreads');
            } elseif ($cmd === 'reopen') {
                if ($this->is_moderator) {
                    foreach ($thread_ids as $thread_id) {
                        $tmp_obj = new ilForumTopic($thread_id);
                        $this->ensureThreadBelongsToForum($this->object->getId(), $tmp_obj);
                        $tmp_obj->reopen();
                    }
                }

                $this->tpl->setOnScreenMessage('success', $this->lng->txt('selected_threads_reopened'), true);
                $this->ctrl->redirect($this, 'showThreads');
            } elseif ($cmd === 'makesticky') {
                if ($this->is_moderator) {
                    $message = $this->lng->txt('sel_threads_make_sticky');

                    foreach ($thread_ids as $thread_id) {
                        $tmp_obj = new ilForumTopic($thread_id);
                        $this->ensureThreadBelongsToForum($this->object->getId(), $tmp_obj);
                        $makeSticky = $tmp_obj->makeSticky();
                        if (!$makeSticky) {
                            $message = $this->lng->txt('sel_threads_already_sticky');
                        }
                    }
                }
                if ($message !== null) {
                    $this->tpl->setOnScreenMessage('info', $message, true);
                }
                $this->ctrl->redirect($this, 'showThreads');
            } elseif ($cmd === 'unmakesticky' || $cmd === 'make_topics_non_sticky') {
                if ($this->is_moderator) {
                    $message = $this->lng->txt('sel_threads_make_unsticky');
                    foreach ($thread_ids as $thread_id) {
                        $tmp_obj = new ilForumTopic($thread_id);
                        $this->ensureThreadBelongsToForum($this->object->getId(), $tmp_obj);
                        $unmakeSticky = $tmp_obj->unmakeSticky();
                        if (!$unmakeSticky) {
                            $message = $this->lng->txt('sel_threads_already_unsticky');
                        }
                    }
                }

                if ($message !== null) {
                    $this->tpl->setOnScreenMessage('info', $message, true);
                }
                $this->ctrl->redirect($this, 'showThreads');
            } elseif ($cmd === 'editThread') {
                if ($this->is_moderator) {
                    $count = count($thread_ids);
                    if ($count !== 1) {
                        $this->tpl->setOnScreenMessage('info', $this->lng->txt('select_max_one_thread'), true);
                        $this->ctrl->redirect($this, 'showThreads');
                    } else {
                        $this->updateThreadObject();
                        return;
                    }
                }

                $this->ctrl->redirect($this, 'showThreads');
            } elseif ($cmd === 'confirmDeleteThreads') {
                $this->confirmDeleteThreadsObject();
            } elseif ($cmd === 'mergeThreads') {
                $this->mergeThreadsObject();
            } else {
                $this->tpl->setOnScreenMessage('info', $this->lng->txt('topics_please_select_one_action'), true);
                $this->ctrl->redirect($this, 'showThreads');
            }
        } else {
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('select_at_least_one_thread'), true);
            $this->ctrl->redirect($this, 'showThreads');
        }
    }

    public function performMoveThreadsObject(): void
    {
        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $frm_ref_id = null;
        if ($this->http->wrapper()->post()->has('frm_ref_id')) {
            $frm_ref_id = $this->http->wrapper()->post()->retrieve(
                'frm_ref_id',
                $this->refinery->kindlyTo()->int()
            );
        } else {
            $this->error->raiseError('Please select a forum', $this->error->MESSAGE);
        }

        $threads2move = ilSession::get('threads2move');
        if (!is_array($threads2move) || $threads2move === []) {
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('select_at_least_one_thread'), true);
            $this->ctrl->redirect($this, 'showThreads');
        }

        if (!$this->access->checkAccess('read', '', (int) $frm_ref_id)) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $threads = [];
        array_walk($threads2move, function (int $threadId) use (&$threads): void {
            $thread = new ilForumTopic($threadId);
            $this->ensureThreadBelongsToForum($this->object->getId(), $thread);

            $threads[] = $threadId;
        });

        if (isset($frm_ref_id) && (int) $frm_ref_id) {
            $errorMessages = $this->object->Forum->moveThreads(
                (array) (ilSession::get('threads2move') ?? []),
                $this->object,
                $this->ilObjDataCache->lookupObjId((int) $frm_ref_id)
            );

            if ([] !== $errorMessages) {
                $this->tpl->setOnScreenMessage('failure', implode('<br><br>', $errorMessages), true);
                $this->ctrl->redirect($this, 'showThreads');
            }

            ilSession::set('threads2move', []);
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('threads_moved_successfully'), true);
            $this->ctrl->redirect($this, 'showThreads');
        } else {
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('no_forum_selected'));
            $this->moveThreadsObject();
        }
    }

    public function cancelMoveThreadsObject(): void
    {
        ilSession::set('threads2move', []);
        $this->ctrl->redirect($this, 'showThreads');
    }

    public function moveThreadsObject(): bool
    {
        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $frm_ref_id = $this->http->wrapper()->post()->retrieve(
            'frm_ref_id',
            $this->refinery->byTrying([$this->refinery->kindlyTo()->int(), $this->refinery->always(null)])
        );

        $threads2move = ilSession::get('threads2move');
        if (!is_array($threads2move) || $threads2move === []) {
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('select_at_least_one_thread'), true);
            $this->ctrl->redirect($this, 'showThreads');
        }

        $threads = [];
        $isModerator = $this->is_moderator;
        array_walk($threads2move, function (int $threadId) use (&$threads, $isModerator): void {
            $thread = new ilForumTopic($threadId, $isModerator);
            $this->ensureThreadBelongsToForum($this->object->getId(), $thread);

            $threads[] = $thread;
        });

        $exp = new ilForumMoveTopicsExplorer($this, 'moveThreads');
        $exp->setPathOpen($this->object->getRefId());
        $exp->setNodeSelected(isset($frm_ref_id) && (int) $frm_ref_id ? (int) $frm_ref_id : 0);
        $exp->setCurrentFrmRefId($this->object->getRefId());
        $exp->setHighlightedNode((string) $this->object->getRefId());
        if (!$exp->handleCommand()) {
            $moveThreadTemplate = new ilTemplate(
                'tpl.forums_threads_move.html',
                true,
                true,
                'components/ILIAS/Forum'
            );

            if (!$this->hideToolbar()) {
                $this->toolbar->addButton($this->lng->txt('back'), $this->ctrl->getLinkTarget($this));
            }

            $tblThr = new ilTable2GUI($this);

            $counter = 0;
            $result = [];
            foreach ($threads as $thread) {
                $result[$counter]['num'] = $counter + 1;
                $result[$counter]['thr_subject'] = $thread->getSubject();
                ++$counter;
            }

            $tblThr->setId('frmthrmv' . $this->object->getRefId());
            $tblThr->setTitle('');
            $tblThr->addColumn($this->lng->txt('subject'), 'top_name', '100%');
            $tblThr->disable('header');
            $tblThr->disable('footer');
            $tblThr->disable('linkbar');
            $tblThr->disable('sort');
            $tblThr->disable('linkbar');
            $tblThr->setLimit(PHP_INT_MAX);
            $tblThr->setRowTemplate('tpl.forums_threads_move_thr_row.html', 'components/ILIAS/Forum');
            $tblThr->setDefaultOrderField('is_sticky');

            $tblThr->setData($result);
            $moveThreadTemplate->setVariable('THREAD_TITLE', sprintf($this->lng->txt('move_chosen_topics'), $thread->getSubject()));
            $moveThreadTemplate->setVariable('THREADS_TABLE', $tblThr->getHTML());
            $moveThreadTemplate->setVariable('FRM_SELECTION_TREE', $exp->getHTML());
            $moveThreadTemplate->setVariable('CMD_SUBMIT', 'performMoveThreads');
            $moveThreadTemplate->setVariable('TXT_SUBMIT', $this->lng->txt('move'));
            $moveThreadTemplate->setVariable('FORMACTION', $this->ctrl->getFormAction($this, 'performMoveThreads'));

            $this->tpl->setContent($moveThreadTemplate->get());
        }

        return true;
    }

    private function isWritingWithPseudonymAllowed(): bool
    {
        return (
            $this->objProperties->isAnonymized() &&
            (!$this->is_moderator || !$this->objProperties->getMarkModeratorPosts())
        );
    }

    protected function deleteThreadDraftsObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $draftIds = array_filter((array) ($this->httpRequest->getParsedBody()['draft_ids'] ?? []));

        $instances = ilForumPostDraft::getDraftInstancesByUserId($this->user->getId());
        $checkedDraftIds = [];
        foreach ($draftIds as $draftId) {
            if (array_key_exists($draftId, $instances)) {
                $checkedDraftIds[] = $draftId;
                $draft = $instances[$draftId];

                $this->deleteMobsOfDraft($draft->getDraftId(), $draft->getPostMessage());

                $GLOBALS['ilAppEventHandler']->raise(
                    'components/ILIAS/Forum',
                    'deletedDraft',
                    [
                        'draftObj' => $draft,
                        'obj_id' => $this->object->getId(),
                        'is_file_upload_allowed' => $this->objProperties->isFileUploadAllowed(),
                    ]
                );

                $draftFileData = new ilFileDataForumDrafts();
                $draftFileData->delete([$draft->getDraftId()]);

                $draft->deleteDraft();
            }
        }

        if (count($checkedDraftIds) > 1) {
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('delete_drafts_successfully'), true);
        } else {
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('delete_draft_successfully'), true);
        }
        $this->ctrl->redirect($this, 'showThreads');
    }

    private function buildThreadForm(bool $isDraft = false): ilForumThreadFormGUI
    {
        $draftId = (int) ($this->httpRequest->getQueryParams()['draft_id'] ?? 0);
        $allowNotification = !$this->objProperties->isAnonymized() && !$this->user->isAnonymous();

        $mail = new ilMail($this->user->getId());
        if (!$this->rbac->system()->checkAccess('internal_mail', $mail->getMailObjectReferenceId())) {
            $allowNotification = false;
        }

        $default_form = new ilForumThreadFormGUI(
            $this,
            $this->objProperties,
            $this->isWritingWithPseudonymAllowed(),
            $allowNotification,
            $isDraft,
            $draftId
        );

        $default_form->addInputItem(ilForumThreadFormGUI::ALIAS_INPUT);
        $default_form->addInputItem(ilForumThreadFormGUI::SUBJECT_INPUT);
        $default_form->addInputItem(ilForumThreadFormGUI::MESSAGE_INPUT);
        $default_form->addInputItem(ilForumThreadFormGUI::FILE_UPLOAD_INPUT);
        $default_form->addInputItem(ilForumThreadFormGUI::ALLOW_NOTIFICATION_INPUT);

        $default_form->generateDefaultForm();

        $this->decorateWithAutosave($default_form);

        return $default_form;
    }

    private function buildMinimalThreadForm(bool $isDraft = false): ilForumThreadFormGUI
    {
        $draftId = (int) ($this->httpRequest->getQueryParams()['draft_id'] ?? 0);
        $allowNotification = !$this->objProperties->isAnonymized() && !$this->user->isAnonymous();

        $mail = new ilMail($this->user->getId());
        if (!$this->rbac->system()->checkAccess('internal_mail', $mail->getMailObjectReferenceId())) {
            $allowNotification = false;
        }

        $minimal_form = new ilForumThreadFormGUI(
            $this,
            $this->objProperties,
            $this->isWritingWithPseudonymAllowed(),
            $allowNotification,
            $isDraft,
            $draftId
        );

        $minimal_form->addInputItem(ilForumThreadFormGUI::ALIAS_INPUT);
        $minimal_form->addInputItem(ilForumThreadFormGUI::SUBJECT_INPUT);

        $minimal_form->generateMinimalForm();

        return $minimal_form;
    }

    private function createThreadObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('add_thread', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $tpl = new ilTemplate('tpl.create_thread_form.html', true, true, 'components/ILIAS/Forum');

        $accordion = new ilAccordionGUI();
        $accordion->setId('acc_' . $this->obj_id);
        $accordion->setBehaviour(ilAccordionGUI::FIRST_OPEN);

        $accordion->addItem($this->lng->txt('new_thread_with_post'), $this->buildThreadForm()->getHTML());
        $accordion->addItem($this->lng->txt('empty_thread'), $this->buildMinimalThreadForm()->getHTML());

        $tpl->setVariable('CREATE_FORM', $accordion->getHTML());
        $tpl->parseCurrentBlock();

        $this->tpl->setContent($tpl->get());
    }

    /**
     * Refactored thread creation to method, refactoring to a separate class should be done in next refactoring steps
     */
    private function createThread(ilForumPostDraft $draft, bool $createFromDraft = false): void
    {
        if (!$this->access->checkAccess('add_thread', '', $this->object->getRefId()) ||
            !$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $frm = $this->object->Forum;
        $frm->setForumId($this->object->getId());
        $frm->setForumRefId($this->object->getRefId());
        $frm->setMDB2WhereCondition('top_frm_fk = %s ', ['integer'], [$frm->getForumId()]);
        $topicData = $frm->getOneTopic();

        $form = $this->buildThreadForm($createFromDraft);
        $minimal_form = $this->buildMinimalThreadForm($createFromDraft);

        if ($form->checkInput()) {
            $userIdForDisplayPurposes = $this->user->getId();
            if ($this->isWritingWithPseudonymAllowed()) {
                $userIdForDisplayPurposes = 0;
            }

            $status = true;
            if (
                ($this->objProperties->isPostActivationEnabled() && !$this->is_moderator) ||
                $this->objCurrentPost->isAnyParentDeactivated()
            ) {
                $status = false;
            }

            if ($createFromDraft) {
                $newThread = new ilForumTopic(0, true, true);
                $newThread->setForumId($topicData->getTopPk());
                $newThread->setThrAuthorId($draft->getPostAuthorId());
                $newThread->setDisplayUserId($draft->getPostDisplayUserId());
                $newThread->setSubject($this->handleFormInput($form->getInput('subject'), false));
                $newThread->setUserAlias($draft->getPostUserAlias());

                $newPost = $frm->generateThread(
                    $newThread,
                    ilRTE::_replaceMediaObjectImageSrc($form->getInput('message')),
                    $draft->isNotificationEnabled() && !$this->user->isAnonymous(),
                    $draft->isPostNotificationEnabled() && !$this->user->isAnonymous(),
                    $status
                );
            } else {
                $newThread = new ilForumTopic(0, true, true);
                $newThread->setForumId($topicData->getTopPk());
                $newThread->setThrAuthorId($this->user->getId());
                $newThread->setDisplayUserId($userIdForDisplayPurposes);
                $newThread->setSubject($this->handleFormInput($form->getInput('subject'), false));
                $newThread->setUserAlias(ilForumUtil::getPublicUserAlias(
                    $form->getInput('alias'),
                    $this->objProperties->isAnonymized()
                ));

                $newPost = $frm->generateThread(
                    $newThread,
                    ilRTE::_replaceMediaObjectImageSrc($form->getInput('message')),
                    $form->getItemByPostVar('notify') && $form->getInput('notify') && !$this->user->isAnonymous(),
                    false, // #19980
                    $status
                );
            }

            $post_obj = new ilForumPost($newPost);
            if ($draft->getRCID() > 0) {
                $post_obj->setRCID($draft->getRCID());
                $post_obj->update();
            }

            if ($this->objProperties->isFileUploadAllowed()) {
                $file = $_FILES['userfile'];
                if (is_array($file) && !empty($file)) {
                    $fileData = new ilFileDataForum($this->object->getId(), $newPost);
                    $fileData->storeUploadedFiles();
                }
            }

            $frm->setDbTable('frm_data');
            $frm->setMDB2WhereCondition('top_pk = %s ', ['integer'], [$topicData->getTopPk()]);
            $frm->updateVisits($topicData->getTopPk());

            if ($createFromDraft) {
                $mediaObjects = ilObjMediaObject::_getMobsOfObject('frm~:html', $this->user->getId());
            } else {
                $mediaObjects = ilRTE::_getMediaObjects($form->getInput('message'));
            }
            foreach ($mediaObjects as $mob) {
                if (ilObjMediaObject::_exists($mob)) {
                    ilObjMediaObject::_removeUsage($mob, 'frm~:html', $this->user->getId());
                    ilObjMediaObject::_saveUsage($mob, 'frm:html', $newPost);
                }
            }

            if ($draft->getDraftId() > 0) {
                $draftHistory = new ilForumDraftsHistory();
                $draftHistory->deleteHistoryByDraftIds([$draft->getDraftId()]);
                $draft->deleteDraft();
            }

            $GLOBALS['ilAppEventHandler']->raise(
                'components/ILIAS/Forum',
                'createdPost',
                [
                    'object' => $this->object,
                    'ref_id' => $this->object->getRefId(),
                    'post' => $post_obj,
                    'notify_moderators' => !$status
                ]
            );

            $this->tpl->setOnScreenMessage('success', $this->lng->txt('forums_thread_new_entry'), true);
            $this->ctrl->redirect($this);
        }

        $form->setValuesByPost();
        if (!$this->objProperties->isAnonymized()) {
            $form->getItemByPostVar('alias')->setValue($this->user->getLogin());
        }

        $accordion = new ilAccordionGUI();
        $accordion->setId('acc_' . $this->obj_id);
        $accordion->setBehaviour(ilAccordionGUI::FIRST_OPEN);
        $accordion->addItem($this->lng->txt('new_thread_with_post'), $form->getHTML());
        $accordion->addItem($this->lng->txt('empty_thread'), $minimal_form->getHTML());

        $this->tpl->setContent($accordion->getHTML());
    }

    /**
     * Refactored thread creation to method, refactoring to a separate class should be done in next refactoring steps
     */
    private function createEmptyThread(): void
    {
        if (!$this->access->checkAccess('add_thread', '', $this->object->getRefId()) ||
            !$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $frm = $this->object->Forum;
        $frm->setForumId($this->object->getId());
        $frm->setForumRefId($this->object->getRefId());
        $frm->setMDB2WhereCondition('top_frm_fk = %s ', ['integer'], [$frm->getForumId()]);
        $topicData = $frm->getOneTopic();

        $form = $this->buildThreadForm();
        $minimal_form = $this->buildMinimalThreadForm();

        if ($minimal_form->checkInput()) {
            $userIdForDisplayPurposes = $this->user->getId();
            if ($this->isWritingWithPseudonymAllowed()) {
                $userIdForDisplayPurposes = 0;
            }

            $status = true;
            if (
                ($this->objProperties->isPostActivationEnabled() && !$this->is_moderator) ||
                $this->objCurrentPost->isAnyParentDeactivated()
            ) {
                $status = false;
            }

            $newThread = new ilForumTopic(0, true, true);
            $newThread->setForumId($topicData->getTopPk());
            $newThread->setThrAuthorId($this->user->getId());
            $newThread->setDisplayUserId($userIdForDisplayPurposes);
            $newThread->setSubject($this->handleFormInput($minimal_form->getInput('subject'), false));
            $newThread->setUserAlias(ilForumUtil::getPublicUserAlias(
                $minimal_form->getInput('alias'),
                $this->objProperties->isAnonymized()
            ));

            $frm->generateThread(
                $newThread,
                '',
                false,
                false, // #19980
                $status,
                false
            );

            $frm->setDbTable('frm_data');
            $frm->setMDB2WhereCondition('top_pk = %s ', ['integer'], [$topicData->getTopPk()]);
            $frm->updateVisits($topicData->getTopPk());

            $this->tpl->setOnScreenMessage('success', $this->lng->txt('forums_thread_new_entry'), true);
            $this->ctrl->redirect($this);
        }

        $form->setValuesByPost();

        if (!$this->objProperties->isAnonymized()) {
            $form->getItemByPostVar('alias')->setValue($this->user->getLogin());
        }

        $accordion = new ilAccordionGUI();
        $accordion->setId('acc_' . $this->obj_id);
        $accordion->setBehaviour(ilAccordionGUI::FIRST_OPEN);
        $accordion->addItem($this->lng->txt('new_thread_with_post'), $form->getHTML());
        $accordion->addItem($this->lng->txt('empty_thread'), $minimal_form->getHTML());

        $this->tpl->setContent($accordion->getHTML());
    }

    protected function publishThreadDraftObject(): void
    {
        if (!ilForumPostDraft::isSavePostDraftAllowed()) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $draft = ilForumPostDraft::newInstanceByDraftId($this->retrieveDraftId());
        $this->checkDraftAccess($draft);

        $this->createThread($draft, true);
    }

    protected function addThreadObject(): void
    {
        $draft = new ilForumPostDraft();
        if (ilForumPostDraft::isSavePostDraftAllowed() && $this->retrieveDraftId() > 0) {
            $draft = ilForumPostDraft::newInstanceByDraftId($this->retrieveDraftId());
            $this->checkDraftAccess($draft);
        }

        $this->createThread($draft);
    }

    protected function addEmptyThreadObject(): void
    {
        $this->createEmptyThread();
    }

    protected function enableForumNotificationObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId()) || $this->user->isAnonymous()) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $frm = $this->object->Forum;
        $frm->setForumId($this->object->getId());
        $frm->enableForumNotification($this->user->getId());

        if ($this->objCurrentTopic->getId() > 0) {
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_forum_notification_enabled'), true);
            $this->ctrl->redirect($this, 'viewThread');
        }

        $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_forum_notification_enabled'));
        $this->showThreadsObject();
    }

    protected function disableForumNotificationObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId()) || $this->user->isAnonymous()) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $frm = $this->object->Forum;
        $frm->setForumId($this->object->getId());
        $frm->disableForumNotification($this->user->getId());

        if ($this->objCurrentTopic->getId() > 0) {
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_forum_notification_disabled'), true);
            $this->ctrl->redirect($this, 'viewThread');
        }

        $this->tpl->setOnScreenMessage('info', $this->lng->txt('forums_forum_notification_disabled'));
        $this->showThreadsObject();
    }

    public function setColumnSettings(ilColumnGUI $column_gui): void
    {
        $column_gui->setBlockProperty('news', 'title', $this->lng->txt('frm_latest_postings'));
        $column_gui->setBlockProperty('news', 'prevent_aggregation', '1');
        $column_gui->setRepositoryMode(true);

        if ($this->access->checkAccess('write', '', $this->object->getRefId())) {
            $news_set = new ilSetting('news');
            if ($news_set->get('enable_rss_for_internal')) {
                $column_gui->setBlockProperty('news', 'settings', '1');
                $column_gui->setBlockProperty('news', 'public_notifications_option', '1');
            }
        }
    }

    protected function addLocatorItems(): void
    {
        if ($this->object instanceof ilObjForum) {
            $this->locator->addItem(
                $this->object->getTitle(),
                $this->ctrl->getLinkTarget($this),
                '',
                $this->object->getRefId()
            );
        }
    }

    public function handleFormInput(string $a_text, bool $a_stripslashes = true): string
    {
        $a_text = str_replace(['<', '>'], ['&lt;', '&gt;'], $a_text);
        if ($a_stripslashes) {
            $a_text = ilUtil::stripSlashes($a_text);
        }

        return $a_text;
    }

    public function prepareFormOutput(string $a_text): string
    {
        $a_text = str_replace(['&lt;', '&gt;'], ['<', '>'], $a_text);

        return ilLegacyFormElementsUtil::prepareFormOutput($a_text);
    }

    protected function infoScreen(): void
    {
        if (
            !$this->access->checkAccess('visible', '', $this->object->getRefId()) &&
            !$this->access->checkAccess('read', '', $this->object->getRefId())
        ) {
            $this->error->raiseError($this->lng->txt('msg_no_perm_read'), $this->error->MESSAGE);
        }

        if (strtolower($this->ctrl->getCmd() ?? '') === 'infoscreen') {
            $this->ctrl->redirectByClass(ilInfoScreenGUI::class, 'showSummary');
        }

        $info = new ilInfoScreenGUI($this);
        $info->enablePrivateNotes();
        $this->ctrl->forwardCommand($info);
    }

    protected function markPostUnreadObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if ($this->objCurrentPost->getId() > 0) {
            $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentPost->getThread());

            $this->object->markPostUnread($this->user->getId(), $this->objCurrentPost->getId());
        }
        $this->viewThreadObject();
    }

    protected function markPostReadObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if ($this->objCurrentTopic->getId() > 0 && $this->objCurrentPost->getId() > 0) {
            $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentPost->getThread());

            $this->object->markPostRead(
                $this->user->getId(),
                $this->objCurrentTopic->getId(),
                $this->objCurrentPost->getId()
            );
        }

        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
        $this->ctrl->redirect($this, 'viewThread');
    }

    protected function initHeaderAction(?string $sub_type = null, ?int $sub_id = null): ?ilObjectListGUI
    {
        $lg = parent::initHeaderAction();

        if (!($lg instanceof ilObjForumListGUI) || !((bool) $this->settings->get('forum_notification', '0'))) {
            return $lg;
        }

        if ($this->user->isAnonymous() || !$this->access->checkAccess('read', '', $this->object->getRefId())) {
            return $lg;
        }

        $frm = $this->object->Forum;
        $frm->setForumId($this->object->getId());
        $frm->setForumRefId($this->object->getRefId());
        $frm->setMDB2Wherecondition('top_frm_fk = %s ', ['integer'], [$frm->getForumId()]);

        $are_notifications_enabled = $frm->isForumNotificationEnabled($this->user->getId());
        $has_membership_enabled_parent_container = $this->object->isParentMembershipEnabledContainer();
        $user_may_disable_notifcations = (
            $this->isUserAllowedToDeactivateNotification() ||
            !$has_membership_enabled_parent_container
        );

        if ($this->objCurrentTopic->getId() > 0) {
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
        }

        if (!$this->user->isAnonymous()) {
            if ($has_membership_enabled_parent_container) {
                if ($are_notifications_enabled && $user_may_disable_notifcations) {
                    $lg->addCustomCommand(
                        $this->ctrl->getLinkTarget($this, 'disableForumNotification'),
                        'forums_disable_forum_notification'
                    );
                } elseif (!$are_notifications_enabled) {
                    $lg->addCustomCommand(
                        $this->ctrl->getLinkTarget($this, 'enableForumNotification'),
                        'forums_enable_forum_notification'
                    );
                }
            } elseif ($are_notifications_enabled) {
                $lg->addCustomCommand(
                    $this->ctrl->getLinkTarget($this, 'disableForumNotification'),
                    'forums_disable_forum_notification'
                );
            } else {
                $lg->addCustomCommand(
                    $this->ctrl->getLinkTarget($this, 'enableForumNotification'),
                    'forums_enable_forum_notification'
                );
            }
        }

        if ($are_notifications_enabled && $user_may_disable_notifcations && !$this->user->isAnonymous()) {
            $frm_noti = new ilForumNotification($this->object->getRefId());
            $frm_noti->setUserId($this->user->getId());
            $interested_events = $frm_noti->readInterestedEvents();

            $events_form_builder = $this->eventsFormBuilder([
                'hidden_value' => '',
                'notify_modified' => (bool) ($interested_events & ilForumNotificationEvents::UPDATED),
                'notify_censored' => (bool) ($interested_events & ilForumNotificationEvents::CENSORED),
                'notify_uncensored' => (bool) ($interested_events & ilForumNotificationEvents::UNCENSORED),
                'notify_post_deleted' => (bool) ($interested_events & ilForumNotificationEvents::POST_DELETED),
                'notify_thread_deleted' => (bool) ($interested_events & ilForumNotificationEvents::THREAD_DELETED),
            ]);

            $notificationsModal = $this->uiFactory->modal()->roundtrip(
                $this->lng->txt('notification_settings'),
                [$events_form_builder->build()]
            )->withActionButtons([
                $this->uiFactory
                    ->button()
                    ->primary($this->lng->txt('save'), '#')
                    ->withOnLoadCode(function (string $id): string {
                        return "
                        (function () {
                          const button = document.getElementById('$id');
                          if (!button) return;
                        
                          const modalDialog = button.closest('.modal-dialog');
                          if (!modalDialog) return;

                          const form = modalDialog.querySelector('.modal-body form');
                          if (!form) return;

                          form.classList.add('ilForumNotificationSettingsForm');
                          button.addEventListener('click', (event) => {
                            event.preventDefault();
                            if (form) {
                              form.submit();
                            }
                          }, true);
                        }());
                        ";
                    })
            ]);

            $showNotificationSettingsBtn = $this->uiFactory->button()
                                                           ->shy($this->lng->txt('notification_settings'), '#')
                                                           ->withOnClick(
                                                               $notificationsModal->getShowSignal()
                                                           );

            $lg->addCustomCommandButton($showNotificationSettingsBtn, $notificationsModal);
        }

        $are_thread_notifications_enabled = false;
        if (!$this->user->isAnonymous() && $this->objCurrentTopic->getId() > 0) {
            $are_thread_notifications_enabled = $this->objCurrentTopic->isNotificationEnabled($this->user->getId());
            if ($are_thread_notifications_enabled) {
                $lg->addCustomCommand(
                    $this->ctrl->getLinkTarget($this, 'toggleThreadNotification'),
                    'forums_disable_notification'
                );
            } else {
                $lg->addCustomCommand(
                    $this->ctrl->getLinkTarget($this, 'toggleThreadNotification'),
                    'forums_enable_notification'
                );
            }
        }
        $this->ctrl->setParameter($this, 'thr_pk', '');

        if (!$this->user->isAnonymous()) {
            if ($are_notifications_enabled || $are_thread_notifications_enabled) {
                $lg->addHeaderIcon(
                    'not_icon',
                    ilUtil::getImagePath('object/notification_on.svg'),
                    $this->lng->txt('frm_notification_activated')
                );
            } else {
                $lg->addHeaderIcon(
                    'not_icon',
                    ilUtil::getImagePath('object/notification_off.svg'),
                    $this->lng->txt('frm_notification_deactivated')
                );
            }
        }

        return $lg;
    }

    /**
     * @param null|array<string, mixed> $predefined_values
     * @throws ilCtrlException
     */
    private function eventsFormBuilder(?array $predefined_values = null): ilForumNotificationEventsFormGUI
    {
        if ($this->objCurrentTopic->getId() > 0) {
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
        }

        return new ilForumNotificationEventsFormGUI(
            $this->ctrl->getFormAction($this, 'saveUserNotificationSettings'),
            $predefined_values,
            $this->uiFactory,
            $this->lng
        );
    }

    public function saveUserNotificationSettingsObject(): void
    {
        $events_form_builder = $this->eventsFormBuilder();

        if ($this->httpRequest->getMethod() === 'POST') {
            $form = $events_form_builder->build()->withRequest($this->httpRequest);
            $formData = $form->getData();

            $interested_events = ilForumNotificationEvents::DEACTIVATED;

            foreach ($events_form_builder->getValidEvents() as $event) {
                $interested_events += isset($formData[$event]) && $formData[$event] ? $events_form_builder->getValueForEvent(
                    $event
                ) : 0;
            }

            $frm_noti = new ilForumNotification($this->object->getRefId());
            $frm_noti->setUserId($this->user->getId());
            $frm_noti->setInterestedEvents($interested_events);
            $frm_noti->updateInterestedEvents();
        }

        $this->tpl->setOnScreenMessage('success', $this->lng->txt('saved_successfully'), true);

        if ($this->objCurrentTopic->getId() > 0) {
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
            $this->ctrl->redirect($this, 'viewThread');
        }

        $this->ctrl->redirect($this, 'showThreads');
    }

    public function isUserAllowedToDeactivateNotification(): bool
    {
        if ($this->objProperties->getNotificationType() === NotificationType::DEFAULT) {
            return true;
        }

        if (!$this->objProperties->isUserToggleNoti() &&
            $this->objProperties->getNotificationType() === NotificationType::ALL_USERS) {
            return true;
        }

        if ($this->objProperties->getNotificationType() === NotificationType::PER_USER &&
            $this->object->isParentMembershipEnabledContainer()) {
            $frm_noti = new ilForumNotification($this->object->getRefId());
            $frm_noti->setUserId($this->user->getId());

            return !$frm_noti->isUserToggleNotification();
        }

        return false;
    }

    public function mergeThreadsObject(): void
    {
        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $threadIdToMerge = $this->http->wrapper()->query()->retrieve(
            'thr_pk',
            $this->refinery->byTrying([
                $this->refinery->kindlyTo()->int(),
                $this->refinery->always(0)
            ])
        );
        if ($threadIdToMerge <= 0) {
            $threadIds = array_values(
                array_filter(array_map(intval(...), (array) ($this->httpRequest->getParsedBody()['thread_ids'] ?? [])))
            );
            if (count($threadIds) === 1) {
                $threadIdToMerge = current($threadIds);
            } else {
                $this->tpl->setOnScreenMessage('info', $this->lng->txt('select_one'));
                $this->showThreadsObject();
                return;
            }
        }

        $frm = $this->object->Forum;
        $frm->setForumId($this->object->getId());
        $frm->setForumRefId($this->object->getRefId());

        $threadToMerge = new ilForumTopic($threadIdToMerge);

        if (ilForum::_lookupObjIdForForumId($threadToMerge->getForumId()) !== $frm->getForumId()) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('not_allowed_to_merge_into_another_forum'));
            $this->showThreadsObject();
            return;
        }

        $frm->setMDB2Wherecondition('top_frm_fk = %s ', ['integer'], [$frm->getForumId()]);

        $threadsTemplate = new ilTemplate(
            'tpl.forums_threads_liste.html',
            true,
            true,
            'components/ILIAS/Forum'
        );

        $topicData = $frm->getOneTopic();
        if ($topicData->getTopPk() > 0) {
            $this->ctrl->setParameter($this, 'merge_thread_id', $threadIdToMerge);
            $tbl = new ilForumTopicTableGUI(
                $this,
                'mergeThreads',
                (int) $this->httpRequest->getQueryParams()['ref_id'],
                $topicData,
                $this->is_moderator
            );
            $tbl->setSelectedThread($threadToMerge);
            $tbl->setMapper($frm)->fetchData();
            $tbl->init();
            $threadsTemplate->setVariable('THREADS_TABLE', $tbl->getHTML());
            $this->tpl->setContent($threadsTemplate->get());
        } else {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('select_one'));
            $this->showThreadsObject();
        }
    }

    public function confirmMergeThreadsObject(): void
    {
        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $sourceThreadId = (int) ($this->httpRequest->getQueryParams()['merge_thread_id'] ?? 0);
        $targetThreadIds = array_values(
            array_filter(array_map(intval(...), (array) ($this->httpRequest->getParsedBody()['thread_ids'] ?? [])))
        );

        if ($sourceThreadId <= 0 || count($targetThreadIds) !== 1) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('select_one'));
            $this->mergeThreadsObject();
            return;
        }

        $targetThreadId = current($targetThreadIds);
        if ($sourceThreadId === $targetThreadId) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('error_same_thread_ids'));
            $this->showThreadsObject();
            return;
        }

        if (ilForumTopic::lookupForumIdByTopicId($sourceThreadId) !== ilForumTopic::lookupForumIdByTopicId($targetThreadId)) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('not_allowed_to_merge_into_another_forum'));
            $this->ctrl->clearParameters($this);
            $this->showThreadsObject();
            return;
        }

        if (ilForumTopic::lookupCreationDate($sourceThreadId) < ilForumTopic::lookupCreationDate($targetThreadId)) {
            $this->tpl->setOnScreenMessage('info', $this->lng->txt('switch_threads_for_merge'));
        }

        $this->ensureThreadBelongsToForum($this->object->getId(), new ilForumTopic($sourceThreadId));
        $this->ensureThreadBelongsToForum($this->object->getId(), new ilForumTopic($targetThreadId));

        $c_gui = new ilConfirmationGUI();

        $c_gui->setFormAction($this->ctrl->getFormAction($this, 'performMergeThreads'));
        $c_gui->setHeaderText($this->lng->txt('frm_sure_merge_threads'));
        $c_gui->setCancel($this->lng->txt('cancel'), 'showThreads');
        $c_gui->setConfirm($this->lng->txt('confirm'), 'performMergeThreads');

        $c_gui->addItem(
            'thread_ids[]',
            (string) $sourceThreadId,
            sprintf($this->lng->txt('frm_merge_src'), ilForumTopic::lookupTitle($sourceThreadId))
        );
        $c_gui->addItem(
            'thread_ids[]',
            (string) $targetThreadId,
            sprintf($this->lng->txt('frm_merge_target'), ilForumTopic::lookupTitle($targetThreadId))
        );
        $this->tpl->setContent($c_gui->getHTML());
    }

    public function performMergeThreadsObject(): void
    {
        if (!$this->is_moderator) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $threadIds = array_values(
            array_filter(array_map(intval(...), (array) ($this->httpRequest->getParsedBody()['thread_ids'] ?? [])))
        );
        if (count($threadIds) !== 2) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('select_one'));
            $this->showThreadsObject();
            return;
        }

        if ((int) $threadIds[0] === (int) $threadIds[1]) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('error_same_thread_ids'));
            $this->showThreadsObject();
            return;
        }

        try {
            $frm = new ilForum();
            $frm->setForumId($this->object->getId());
            $frm->setForumRefId($this->object->getRefId());

            $this->ensureThreadBelongsToForum($this->object->getId(), new ilForumTopic((int) $threadIds[0]));
            $this->ensureThreadBelongsToForum($this->object->getId(), new ilForumTopic((int) $threadIds[1]));

            $frm->mergeThreads((int) $threadIds[0], (int) $threadIds[1]);
            $this->tpl->setOnScreenMessage('success', $this->lng->txt('merged_threads_successfully'));
        } catch (ilException $e) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt($e->getMessage()));
        }

        $this->showThreadsObject();
    }

    protected function setSideBlocks(): void
    {
        $content = $this->getRightColumnHTML();
        if (!$this->ctrl->isAsynch()) {
            $content = implode('', [
                ilRepositoryObjectSearchGUI::getSearchBlockHTML($this->lng->txt('frm_search')),
                $content,
            ]);
        }
        $this->tpl->setRightContent($content);
    }

    protected function deliverDraftZipFileObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $draft = ilForumPostDraft::newInstanceByDraftId($this->retrieveDraftId());
        $this->checkDraftAccess($draft);
        $fileData = new ilFileDataForumDrafts(0, $draft->getDraftId());
        if (!$fileData->deliverZipFile()) {
            $this->ctrl->redirect($this);
        }
    }

    protected function deliverZipFileObject(): void
    {
        if (!$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentPost->getThread());

        $fileData = new ilFileDataForum($this->object->getId(), $this->objCurrentPost->getId());
        if (!$fileData->deliverZipFile()) {
            $this->ctrl->redirect($this);
        }
    }

    /**
     * @param ilPropertyFormGUI|null $form
     */
    protected function editThreadDraftObject(ilPropertyFormGUI $form = null): void
    {
        if (!ilForumPostDraft::isSavePostDraftAllowed() ||
            !$this->access->checkAccess('add_thread', '', $this->object->getRefId()) ||
            !$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $frm = $this->object->Forum;
        $frm->setForumId($this->object->getId());
        $frm->setForumRefId($this->object->getRefId());

        $draft = new ilForumPostDraft();
        $draftId = (int) ($this->httpRequest->getQueryParams()['draft_id'] ?? 0);
        if ($draftId > 0) {
            $draft = ilForumPostDraft::newInstanceByDraftId($draftId);
            $this->checkDraftAccess($draft);
        }

        $do_history_check = (bool) ($this->httpRequest->getQueryParams()['hist_check'] ?? true);
        if (!($form instanceof ilPropertyFormGUI) && $do_history_check) {
            $this->doHistoryCheck($draft->getDraftId());
        }

        if ($form instanceof ilPropertyFormGUI) {
            $this->ctrl->setParameter($this, 'draft_id', $draftId);
        } else {
            $form = $this->buildThreadForm(true);
            $form->setValuesByArray([
                'alias' => $draft->getPostUserAlias(),
                'subject' => $draft->getPostSubject(),
                'message' => ilRTE::_replaceMediaObjectImageSrc($frm->prepareText($draft->getPostMessage(), 2), 1),
                'notify' => $draft->isNotificationEnabled() && !$this->user->isAnonymous(),
                'userfile' => '',
                'del_file' => [],
                'draft_id' => $draftId
            ]);
        }

        $this->tpl->setContent($form->getHTML() . $this->modal_history);
    }

    protected function restoreFromHistoryObject(): void
    {
        if (!ilForumPostDraft::isSavePostDraftAllowed()) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $historyId = (int) ($this->httpRequest->getQueryParams()['history_id'] ?? 0);
        $history = new ilForumDraftsHistory($historyId);

        $this->checkDraftAccess($history->getDraftId());

        $draft = $history->rollbackAutosave();
        if ($draft->getThreadId() === 0 && $draft->getPostId() === 0) {
            $this->ctrl->setParameter($this, 'draft_id', $history->getDraftId());
            $this->ctrl->redirect($this, 'editThreadDraft');
        }

        $this->ctrl->clearParameters($this);
        $this->ctrl->setParameter($this, 'pos_pk', $draft->getPostId());
        $this->ctrl->setParameter($this, 'thr_pk', $draft->getThreadId());
        $this->ctrl->setParameter($this, 'draft_id', $draft->getDraftId());
        $this->ctrl->setParameter($this, 'action', 'editdraft');

        ilForumPostDraft::createDraftBackup($draft->getDraftId());

        $this->ctrl->redirect($this, 'viewThread');
    }

    protected function saveThreadAsDraftObject(): void
    {
        if (!ilForumPostDraft::isSavePostDraftAllowed() ||
            !$this->access->checkAccess('add_thread', '', $this->object->getRefId()) ||
            !$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $autosave_draft_id = (int) ($this->httpRequest->getParsedBody()['draft_id'] ?? 0);
        if ($autosave_draft_id <= 0) {
            $autosave_draft_id = (int) ($this->httpRequest->getQueryParams()['draft_id'] ?? 0);
        }

        $frm = $this->object->Forum;
        $frm->setForumId($this->object->getId());
        $frm->setForumRefId($this->object->getRefId());
        $frm->setMDB2WhereCondition('top_frm_fk = %s ', ['integer'], [$frm->getForumId()]);
        $topicData = $frm->getOneTopic();

        $form = $this->buildThreadForm();
        if ($form->checkInput()) {
            if ($autosave_draft_id === 0) {
                $draft = new ilForumPostDraft();
                $draft->setForumId($topicData->getTopPk());
                $draft->setThreadId(0);
                $draft->setPostId(0);
            } else {
                $draft = ilForumPostDraft::newInstanceByDraftId($autosave_draft_id);
                $this->checkDraftAccess($draft);
            }

            $draft->setPostSubject($this->handleFormInput($form->getInput('subject'), false));
            $draft->setPostMessage(ilRTE::_replaceMediaObjectImageSrc($form->getInput('message')));
            $draft->setPostUserAlias(ilForumUtil::getPublicUserAlias(
                $form->getInput('alias'),
                $this->objProperties->isAnonymized()
            ));
            $draft->setNotificationStatus($form->getInput('notify') && !$this->user->isAnonymous());
            $draft->setPostAuthorId($this->user->getId());
            $draft->setPostDisplayUserId($this->isWritingWithPseudonymAllowed() ? 0 : $this->user->getId());

            if ($autosave_draft_id === 0) {
                $draftId = $draft->saveDraft();
            } else {
                $draft->updateDraft();
                $draftId = $draft->getDraftId();
            }

            $GLOBALS['ilAppEventHandler']->raise(
                'components/ILIAS/Forum',
                'savedAsDraft',
                [
                    'draftObj' => $draft,
                    'obj_id' => $this->object->getId(),
                    'is_file_upload_allowed' => $this->objProperties->isFileUploadAllowed(),
                ]
            );

            ilForumUtil::moveMediaObjects($form->getInput('message'), 'frm~d:html', $draftId, 'frm~d:html', $draftId);

            $draftFileData = new ilFileDataForumDrafts($this->object->getId(), $draftId);

            $files2delete = $form->getInput('del_file');
            if (is_array($files2delete) && $files2delete !== []) {
                $draftFileData->unlinkFilesByMD5Filenames($files2delete);
            }

            if ($this->objProperties->isFileUploadAllowed()) {
                $file = $_FILES['userfile'];
                if (is_array($file) && !empty($file)) {
                    $draftFileData->storeUploadedFiles();
                }
            }

            $this->tpl->setOnScreenMessage('success', $this->lng->txt('save_draft_successfully'), true);
            $this->ctrl->clearParameters($this);
            $this->ctrl->redirect($this, 'showThreads');
        }

        $this->requestAction = substr($this->requestAction, 6);
        $form->setValuesByPost();
        $this->ctrl->setParameter($this, 'draft_id', $autosave_draft_id);
        $this->tpl->setContent($form->getHTML());
    }

    protected function updateThreadDraftObject(): void
    {
        if (!ilForumPostDraft::isSavePostDraftAllowed() ||
            !$this->access->checkAccess('add_thread', '', $this->object->getRefId()) ||
            !$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $draft = ilForumPostDraft::newInstanceByDraftId($this->retrieveDraftId());
        $this->checkDraftAccess($draft);

        $form = $this->buildThreadForm(true);
        if ($form->checkInput()) {
            $draft->setPostSubject($this->handleFormInput($form->getInput('subject'), false));
            $draft->setPostMessage(ilRTE::_replaceMediaObjectImageSrc($form->getInput('message')));
            $draft->setPostUserAlias(ilForumUtil::getPublicUserAlias(
                $form->getInput('alias'),
                $this->objProperties->isAnonymized()
            ));
            $draft->setNotificationStatus($form->getInput('notify') && !$this->user->isAnonymous());
            $draft->setPostAuthorId($this->user->getId());
            $draft->setPostDisplayUserId($this->isWritingWithPseudonymAllowed() ? 0 : $this->user->getId());
            $draft->updateDraft();

            $GLOBALS['ilAppEventHandler']->raise(
                'components/ILIAS/Forum',
                'updatedDraft',
                [
                    'draftObj' => $draft,
                    'obj_id' => $this->object->getId(),
                    'is_file_upload_allowed' => $this->objProperties->isFileUploadAllowed(),
                ]
            );

            ilForumUtil::moveMediaObjects(
                $form->getInput('message'),
                'frm~d:html',
                $draft->getDraftId(),
                'frm~d:html',
                $draft->getDraftId()
            );

            $draftFileData = new ilFileDataForumDrafts($this->object->getId(), $draft->getDraftId());

            $files2delete = $form->getInput('del_file');
            if (is_array($files2delete) && $files2delete !== []) {
                $draftFileData->unlinkFilesByMD5Filenames($files2delete);
            }

            if ($this->objProperties->isFileUploadAllowed()) {
                $file = $_FILES['userfile'];
                if (is_array($file) && !empty($file) && isset($file['full_path'][0]) && !empty($file['full_path'][0])) {
                    $draftFileData->storeUploadedFiles();
                }
            }

            $this->tpl->setOnScreenMessage('success', $this->lng->txt('save_draft_successfully'), true);
            $this->ctrl->clearParameters($this);
            $this->ctrl->redirect($this, 'showThreads');
        }

        $form->setValuesByPost();
        $this->ctrl->setParameter($this, 'hist_check', 0);
        $this->ctrl->setParameter($this, 'draft_id', $draft->getDraftId());
        $this->editThreadDraftObject($form);
    }

    public function saveTopLevelDraftObject(): void
    {
        $this->saveAsDraftObject();
    }

    public function saveAsDraftObject(): void
    {
        if (!ilForumPostDraft::isSavePostDraftAllowed() ||
            !$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $autosave_draft_id = $this->http->wrapper()->post()->retrieve(
            'draft_id',
            $this->refinery->byTrying([
                $this->refinery->kindlyTo()->int(),
                $this->refinery->always(null)
            ])
        );

        if ($this->objCurrentTopic->getId() === 0) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_thr_deleted'), true);
            $this->ctrl->redirect($this);
        }

        if ($this->objCurrentTopic->isClosed()) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_thr_closed'), true);
            $this->ctrl->redirect($this);
        }

        $oReplyEditForm = $this->getReplyEditForm();
        if ($oReplyEditForm->checkInput()) {
            if ($this->objCurrentPost->getId() === 0) {
                $this->requestAction = '';
                $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_parent_deleted'), true);
                $this->viewThreadObject();
                return;
            }

            $oForumObjects = $this->getForumObjects();
            $frm = $oForumObjects['frm'];
            $frm->setMDB2WhereCondition(' top_frm_fk = %s ', ['integer'], [$frm->getForumId()]);
            $topicData = $frm->getOneTopic();

            if ($this->requestAction === 'ready_showreply') {
                if (!$this->access->checkAccess('add_reply', '', $this->object->getRefId())) {
                    $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
                }

                if ($autosave_draft_id > 0) {
                    $draftObj = ilForumPostDraft::newInstanceByDraftId($autosave_draft_id);
                    $this->checkDraftAccess($draftObj->getDraftId());
                } else {
                    $draftObj = new ilForumPostDraft();
                    $this->ensureThreadBelongsToForum($this->object->getId(), $this->objCurrentPost->getThread());
                    $draftObj->setForumId($topicData->getTopPk());
                    $draftObj->setThreadId($this->objCurrentTopic->getId());
                    $draftObj->setPostId($this->objCurrentPost->getId());
                }

                $draftObj->setPostSubject($this->handleFormInput($oReplyEditForm->getInput('subject'), false));
                $draftObj->setPostMessage(ilRTE::_replaceMediaObjectImageSrc($oReplyEditForm->getInput('message')));
                $draftObj->setPostUserAlias(ilForumUtil::getPublicUserAlias(
                    $oReplyEditForm->getInput('alias'),
                    $this->objProperties->isAnonymized()
                ));
                $draftObj->setNotificationStatus($oReplyEditForm->getInput('notify') && !$this->user->isAnonymous());
                $draftObj->setPostNotificationStatus($oReplyEditForm->getInput('notify_post') && !$this->user->isAnonymous());

                $draftObj->setPostAuthorId($this->user->getId());
                $draftObj->setPostDisplayUserId(($this->isWritingWithPseudonymAllowed() ? 0 : $this->user->getId()));

                if ($autosave_draft_id === 0) {
                    $draft_id = $draftObj->saveDraft();
                } else {
                    $draftObj->updateDraft();
                    $draft_id = $draftObj->getDraftId();
                }

                // copy temporary media objects (frm~)
                ilForumUtil::moveMediaObjects(
                    $oReplyEditForm->getInput('message'),
                    'frm~d:html',
                    $draft_id,
                    'frm~d:html',
                    $draft_id
                );

                if ($this->objProperties->isFileUploadAllowed()) {
                    $file = $_FILES['userfile'];
                    if (is_array($file) && !empty($file)) {
                        $oFDForumDrafts = new ilFileDataForumDrafts($this->object->getId(), $draftObj->getDraftId());
                        $oFDForumDrafts->storeUploadedFiles();
                    }
                }

                $GLOBALS['ilAppEventHandler']->raise(
                    'components/ILIAS/Forum',
                    'savedAsDraft',
                    [
                        'draftObj' => $draftObj,
                        'obj_id' => $this->object->getId(),
                        'is_file_upload_allowed' => $this->objProperties->isFileUploadAllowed()
                    ]
                );

                $frm_session_values = ilSession::get('frm');
                if (is_array($frm_session_values)) {
                    $frm_session_values[$this->objCurrentPost->getThreadId()]['openTreeNodes'][] = $this->objCurrentPost->getId();
                }
                ilSession::set('frm', $frm_session_values);

                $this->tpl->setOnScreenMessage('success', $this->lng->txt('save_draft_successfully'), true);
                $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
                $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
                $this->ctrl->redirect($this, 'viewThread');
            }
        } else {
            $oReplyEditForm->setValuesByPost();
            $this->requestAction = substr($this->requestAction, 6);
        }
        $this->viewThreadObject();
    }

    protected function editDraftObject(): void
    {
        $this->doHistoryCheck($this->retrieveDraftId());
        $this->viewThreadObject();
    }

    public function updateDraftObject(): void
    {
        if (!ilForumPostDraft::isSavePostDraftAllowed() ||
            !$this->access->checkAccess('read', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        if ($this->objCurrentTopic->getId() === 0) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_thr_deleted'), true);
            $this->ctrl->redirect($this);
        }

        if ($this->objCurrentTopic->isClosed()) {
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_thr_closed'), true);
            $this->ctrl->redirect($this);
        }

        if ($this->objCurrentPost->getId() === 0) {
            $this->requestAction = '';
            $this->tpl->setOnScreenMessage('failure', $this->lng->txt('frm_action_not_possible_parent_deleted'));
            $this->viewThreadObject();
            return;
        }

        $oReplyEditForm = $this->getReplyEditForm();
        if ($oReplyEditForm->checkInput()) {
            $oForumObjects = $this->getForumObjects();
            $forumObj = $oForumObjects['forumObj'];

            if (!$this->user->isAnonymous() && in_array($this->requestAction, ['showdraft', 'editdraft'])) {
                if (!$this->access->checkAccess('add_reply', '', $this->object->getRefId())) {
                    $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
                }

                $draft = new ilForumPostDraft(
                    $this->user->getId(),
                    $this->objCurrentPost->getId(),
                    $this->retrieveDraftId()
                );

                $this->checkDraftAccess($draft);

                $draft->setPostSubject($this->handleFormInput($oReplyEditForm->getInput('subject'), false));
                $draft->setPostMessage(ilRTE::_replaceMediaObjectImageSrc(
                    $oReplyEditForm->getInput('message')
                ));
                $draft->setPostUserAlias(ilForumUtil::getPublicUserAlias(
                    $oReplyEditForm->getInput('alias'),
                    $this->objProperties->isAnonymized()
                ));
                $draft->setNotificationStatus($oReplyEditForm->getInput('notify') && !$this->user->isAnonymous());
                $draft->setUpdateUserId($this->user->getId());
                $draft->setPostAuthorId($this->user->getId());
                $draft->setPostDisplayUserId($this->isWritingWithPseudonymAllowed() ? 0 : $this->user->getId());

                $draft->updateDraft();

                $GLOBALS['ilAppEventHandler']->raise(
                    'components/ILIAS/Forum',
                    'updatedDraft',
                    [
                        'draftObj' => $draft,
                        'obj_id' => $this->object->getId(),
                        'is_file_upload_allowed' => $this->objProperties->isFileUploadAllowed()
                    ]
                );

                $uploadedObjects = ilObjMediaObject::_getMobsOfObject('frm~:html', $this->user->getId());

                foreach ($uploadedObjects as $mob) {
                    ilObjMediaObject::_removeUsage($mob, 'frm~:html', $this->user->getId());
                    ilObjMediaObject::_saveUsage($mob, 'frm~d:html', $draft->getDraftId());
                }
                ilForumUtil::saveMediaObjects(
                    $oReplyEditForm->getInput('message'),
                    'frm~d:html',
                    $draft->getDraftId()
                );

                $oFDForumDrafts = new ilFileDataForumDrafts($forumObj->getId(), $draft->getDraftId());

                $file2delete = $oReplyEditForm->getInput('del_file');
                if (is_array($file2delete) && count($file2delete)) {
                    $oFDForumDrafts->unlinkFilesByMD5Filenames($file2delete);
                }

                if ($this->objProperties->isFileUploadAllowed()) {
                    $file = $_FILES['userfile'];
                    if (is_array($file) && !empty($file)) {
                        $oFDForumDrafts->storeUploadedFiles();
                    }
                }

                $frm_session_values = ilSession::get('frm');
                if (is_array($frm_session_values)) {
                    $frm_session_values[$this->objCurrentPost->getThreadId()]['openTreeNodes'][] = $this->objCurrentPost->getId();
                }
                ilSession::set('frm', $frm_session_values);
                $this->tpl->setOnScreenMessage('success', $this->lng->txt('save_draft_successfully'), true);
                $this->ctrl->clearParameters($this);
                $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
                $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
                $this->ctrl->setParameter($this, 'draft_id', $draft->getDraftId());
            }
        } else {
            $this->ctrl->clearParameters($this);
            $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
            $this->ctrl->setParameter($this, 'draft_id', $this->retrieveDraftId());
            $this->ctrl->setParameter($this, 'action', 'editdraft');
            $oReplyEditForm->setValuesByPost();
            $this->viewThreadObject();
            return;
        }
        $this->ctrl->clearParameters($this);
        $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
        $this->ctrl->redirect($this, 'viewThread');
    }

    protected function deleteMobsOfDraft(int $draft_id, string $message): void
    {
        $oldMediaObjects = ilObjMediaObject::_getMobsOfObject('frm~d:html', $draft_id);
        $curMediaObjects = ilRTE::_getMediaObjects($message);
        foreach ($oldMediaObjects as $oldMob) {
            $found = false;
            foreach ($curMediaObjects as $curMob) {
                if ($oldMob === $curMob) {
                    $found = true;
                    break;
                }
            }
            if (!$found && ilObjMediaObject::_exists($oldMob)) {
                ilObjMediaObject::_removeUsage($oldMob, 'frm~d:html', $draft_id);
                $mob_obj = new ilObjMediaObject($oldMob);
                $mob_obj->delete();
            }
        }
    }

    protected function deleteSelectedDraft(): void
    {
        $draft = new ilForumPostDraft($this->user->getId(), $this->objCurrentPost->getId(), $this->retrieveDraftId());
        $this->checkDraftAccess($draft);

        $this->deleteMobsOfDraft($draft->getDraftId(), $draft->getPostMessage());

        $objFileDataForumDrafts = new ilFileDataForumDrafts($this->object->getId());
        $objFileDataForumDrafts->delete([$draft->getDraftId()]);

        $GLOBALS['ilAppEventHandler']->raise(
            'components/ILIAS/Forum',
            'deletedDraft',
            [
                'draftObj' => $draft,
                'obj_id' => $this->object->getId(),
                'is_file_upload_allowed' => $this->objProperties->isFileUploadAllowed()
            ]
        );
        $draft->deleteDraft();

        $this->tpl->setOnScreenMessage('success', $this->lng->txt('delete_draft_successfully'), true);
        $this->ctrl->clearParameters($this);
        $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
        $this->ctrl->redirect($this, 'viewThread');
    }

    protected function autosaveDraftAsyncObject(): void
    {
        if ($this->requestAction !== 'ready_showreply' &&
            ilForumPostDraft::isSavePostDraftAllowed() &&
            $this->access->checkAccess('read', '', $this->object->getRefId()) &&
            $this->access->checkAccess('add_reply', '', $this->object->getRefId())) {
            $action = new ilForumAutoSaveAsyncDraftAction(
                $this->user,
                $this->getReplyEditForm(),
                $this->objProperties,
                $this->objCurrentTopic,
                $this->objCurrentPost,
                function (string $message): string {
                    return $this->handleFormInput($message);
                },
                $this->retrieveDraftId(),
                ilObjForum::lookupForumIdByRefId($this->ref_id),
                ilUtil::stripSlashes($this->requestAction)
            );

            $this->http->saveResponse($this->http->response()->withBody(
                \ILIAS\Filesystem\Stream\Streams::ofString(json_encode(
                    $action->executeAndGetResponseObject(),
                    JSON_THROW_ON_ERROR
                ))
            ));
        }

        $this->http->sendResponse();
        $this->http->close();
    }

    protected function autosaveThreadDraftAsyncObject(): void
    {
        if ($this->requestAction !== 'ready_showreply' &&
            ilForumPostDraft::isSavePostDraftAllowed() &&
            $this->access->checkAccess('read', '', $this->object->getRefId()) &&
            $this->access->checkAccess('add_thread', '', $this->object->getRefId())) {
            $action = new ilForumAutoSaveAsyncDraftAction(
                $this->user,
                $this->buildThreadForm(),
                $this->objProperties,
                $this->objCurrentTopic,
                $this->objCurrentPost,
                function (string $message): string {
                    return $this->handleFormInput($message, false);
                },
                $this->retrieveDraftId(),
                ilObjForum::lookupForumIdByRefId($this->ref_id),
                ilUtil::stripSlashes($this->requestAction)
            );

            $this->http->saveResponse($this->http->response()->withBody(
                \ILIAS\Filesystem\Stream\Streams::ofString(json_encode(
                    $action->executeAndGetResponseObject(),
                    JSON_THROW_ON_ERROR
                ))
            ));
        }

        $this->http->sendResponse();
        $this->http->close();
    }

    private function renderSplitButton(
        ilTemplate $tpl,
        string $action,
        bool $is_post,
        ilForumPost $node,
        int $pageIndex = 0,
        ilForumPostDraft $draft = null
    ): void {
        $draft_id = $this->retrieveDraftId();

        $actions = [];
        if ($is_post) {
            if (($this->objCurrentPost->getId() !== $node->getId() || (
                !in_array($action, ['showreply', 'showedit', 'censor', 'delete'], true) &&
                !$this->displayConfirmPostActivation()
            )) && ($this->is_moderator || $node->isActivated() || $node->isOwner($this->user->getId()))) {
                if ($this->is_moderator && !$this->objCurrentTopic->isClosed() && !$node->isActivated()) {
                    $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
                    $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
                    $this->ctrl->setParameter($this, 'page', $pageIndex);
                    $this->ctrl->setParameter(
                        $this,
                        'orderby',
                        $this->getOrderByParam()
                    );
                    $primary_action = $this->ctrl->getLinkTarget(
                        $this,
                        'askForPostActivation',
                        (string) $node->getId()
                    );
                    $primary_action_language_id = 'activate_post';
                    $this->ctrl->clearParameters($this);
                }
                if (
                    !$this->objCurrentTopic->isClosed() && $node->isActivated() && !$node->isCensored() &&
                    $this->access->checkAccess('add_reply', '', $this->object->getRefId())
                ) {
                    $this->ctrl->setParameter($this, 'action', 'showreply');
                    $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
                    $this->ctrl->setParameter($this, 'page', $pageIndex);
                    $this->ctrl->setParameter(
                        $this,
                        'orderby',
                        $this->getOrderByParam()
                    );
                    $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
                    $primary_action = $this->ctrl->getLinkTarget(
                        $this,
                        'viewThread',
                        'reply_' . $node->getId()
                    );
                    $primary_action_language_id = 'reply_to_postings';
                    $this->ctrl->clearParameters($this);
                }
                if (
                    !$this->objCurrentTopic->isClosed() &&
                    !$node->isCensored() &&
                    !$this->user->isAnonymous() &&
                    ($node->isOwner($this->user->getId()) || $this->is_moderator)
                ) {
                    $this->ctrl->setParameter($this, 'action', 'showedit');
                    $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
                    $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
                    $this->ctrl->setParameter($this, 'page', $pageIndex);
                    $this->ctrl->setParameter(
                        $this,
                        'orderby',
                        $this->getOrderByParam()
                    );
                    $actions['edit'] = $this->ctrl->getLinkTarget($this, 'viewThread', (string) $node->getId());
                    $this->ctrl->clearParameters($this);
                }
                if (!$this->user->isAnonymous()) {
                    $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
                    $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
                    $this->ctrl->setParameter($this, 'page', $pageIndex);
                    $this->ctrl->setParameter(
                        $this,
                        'orderby',
                        $this->getOrderByParam()
                    );
                    $this->ctrl->setParameter($this, 'viewmode', $this->selectedSorting);

                    $read_undread_txt = 'frm_mark_as_read';
                    $read_undread_cmd = 'markPostRead';
                    if ($node->isPostRead()) {
                        $read_undread_txt = 'frm_mark_as_unread';
                        $read_undread_cmd = 'markPostUnread';
                    }
                    $actions[$read_undread_txt] = $this->ctrl->getLinkTarget(
                        $this,
                        $read_undread_cmd,
                        (string) $node->getId()
                    );

                    $this->ctrl->clearParameters($this);
                }
                if (!$node->isCensored()) {
                    $this->ctrl->setParameterByClass(ilForumExportGUI::class, 'top_pk', $node->getForumId());
                    $this->ctrl->setParameterByClass(ilForumExportGUI::class, 'thr_pk', $node->getThreadId());


                    $this->ctrl->clearParameters($this);
                }
                if (
                    !$this->objCurrentTopic->isClosed() &&
                    !$this->user->isAnonymous() &&
                    ($this->is_moderator || ($node->isOwner($this->user->getId()) && !$node->hasReplies()))
                ) {
                    $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
                    $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
                    $this->ctrl->setParameter($this, 'page', $pageIndex);
                    $this->ctrl->setParameter(
                        $this,
                        'orderby',
                        $this->getOrderByParam()
                    );
                    $actions['delete'] = $this->ctrl->getFormAction($this, 'deletePosting');
                    $this->ctrl->clearParameters($this);
                }
                if ($this->is_moderator && !$this->objCurrentTopic->isClosed()) {
                    $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
                    $this->ctrl->setParameter($this, 'thr_pk', $node->getThreadId());
                    $this->ctrl->setParameter($this, 'page', $pageIndex);
                    $this->ctrl->setParameter(
                        $this,
                        'orderby',
                        $this->getOrderByParam()
                    );
                    if ($node->isCensored()) {
                        $this->ctrl->setParameter($this, 'action', 'viewThread');
                        $actions['frm_revoke_censorship'] = $this->ctrl->getFormAction($this, 'revokeCensorship');
                    } else {
                        $actions['frm_censorship'] = $this->ctrl->getFormAction($this, 'addCensorship');
                    }
                    $this->ctrl->clearParameters($this);
                }
            }
        } elseif ($draft_id !== $draft->getDraftId() || !in_array($action, ['deletedraft', 'editdraft'])) {
            // get actions for drafts
            $this->ctrl->setParameter($this, 'action', 'publishdraft');
            $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
            $this->ctrl->setParameter($this, 'page', $pageIndex);
            $this->ctrl->setParameter($this, 'draft_id', $draft->getDraftId());
            $this->ctrl->setParameter(
                $this,
                'orderby',
                $this->getOrderByParam()
            );
            $primary_action = $this->ctrl->getLinkTarget($this, 'publishSelectedDraft', (string) $node->getId());
            $primary_action_language_id = 'publish';
            $this->ctrl->clearParameters($this);

            $this->ctrl->setParameter($this, 'action', 'editdraft');
            $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
            $this->ctrl->setParameter($this, 'draft_id', $draft->getDraftId());
            $this->ctrl->setParameter($this, 'page', $pageIndex);
            $this->ctrl->setParameter(
                $this,
                'orderby',
                $this->getOrderByParam()
            );
            $actions['edit'] = $this->ctrl->getLinkTarget($this, 'editDraft', 'draft_edit_' . $draft->getDraftId());
            $this->ctrl->clearParameters($this);

            $this->ctrl->setParameter($this, 'pos_pk', $node->getId());
            $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());
            $this->ctrl->setParameter($this, 'draft_id', $draft->getDraftId());
            $this->ctrl->setParameter($this, 'page', $pageIndex);
            $this->ctrl->setParameter(
                $this,
                'orderby',
                $this->getOrderByParam()
            );
            $actions['delete'] = $this->ctrl->getFormAction($this, 'deletePostingDraft');
            $this->ctrl->clearParameters($this);

            if ($draft_id !== 0 && $action === 'editdraft') {
                $actions = [];
            }
        }

        $tpl->setCurrentBlock('posts_row');
        if ($actions !== [] && !$this->objCurrentTopic->isClosed()) {
            $items = [];
            foreach ($actions as $lng_id => $url) {
                if ($lng_id === 'frm_revoke_censorship' || $lng_id === 'frm_censorship') {
                    $modalTemplate = new ilTemplate('tpl.forums_censor_modal.html', true, true, 'components/ILIAS/Forum');
                    $form_id = str_replace('.', '_', uniqid('form', true));
                    $modalTemplate->setVariable('FORM_ID', $form_id);

                    if ($node->isCensored()) {
                        $modalTemplate->setVariable('BODY', $this->lng->txt('forums_info_censor2_post'));
                    } else {
                        $modalTemplate->setVariable('BODY', $this->lng->txt('forums_info_censor_post'));
                        $modalTemplate->touchBlock('message');
                    }

                    $modalTemplate->setVariable('FORM_ACTION', $url);

                    $content = $this->uiFactory->legacy($modalTemplate->get());
                    $submitBtn = $this->uiFactory->button()->primary(
                        $this->lng->txt('submit'),
                        '#'
                    )->withOnLoadCode(
                        static function (string $id) use ($form_id): string {
                            return "
                            (function () {
                              const button = document.getElementById('$id');
                              if (!button) return;
                            
                              const form = document.getElementById('$form_id');
                              if (!form) return;
                            
                              button.addEventListener('click', (event) => {
                                event.preventDefault();
                                form.submit();
                              }, true);
                            }());
                            ";
                        }
                    );
                    $modal = $this->uiFactory->modal()->roundtrip(
                        $this->lng->txt($lng_id),
                        [$content]
                    )->withActionButtons([$submitBtn]);
                    $items[] = $this->uiFactory->button()->shy($this->lng->txt($lng_id), '#')->withOnClick(
                        $modal->getShowSignal()
                    );

                    $this->modalActionsContainer[] = $modal;
                    continue;
                }

                if ($lng_id === 'delete') {
                    $modal = $this->uiFactory->modal()->interruptive(
                        $this->lng->txt($lng_id),
                        str_contains($url, 'deletePostingDraft') ?
                            $this->lng->txt('forums_info_delete_draft') :
                            $this->lng->txt('forums_info_delete_post'),
                        $url
                    )->withActionButtonLabel(
                        $this->lng->txt(str_contains($url, 'deletePostingDraft') ? 'deletePostingDraft' : 'deletePosting')
                    );

                    $items[] = $this->uiFactory->button()->shy($this->lng->txt($lng_id), '#')->withOnClick(
                        $modal->getShowSignal()
                    );

                    $this->modalActionsContainer[] = $modal;


                    continue;
                }

                $items[] = $this->uiFactory->button()->shy($this->lng->txt($lng_id), $url);
            }

            $dropdown = $this->uiFactory->dropdown()->standard($items);
            $render_action_buttons = [$dropdown];
            if (isset($primary_action, $primary_action_language_id)) {
                if ($primary_action_language_id === 'activate_post') {
                    $action_button = $this->addActivationFormModal($node);
                } else {
                    $action_button = $this->uiFactory->button()->standard(
                        $this->lng->txt($primary_action_language_id),
                        $primary_action
                    );
                }
                $tpl->setVariable('MAIN_ACTION', $this->uiRenderer->render($action_button));
            }
            $tpl->setVariable('DROPDOWN_ACTIONS', $this->uiRenderer->render($render_action_buttons));
        }
    }

    private function checkDraftAccess(int|ilForumPostDraft $draft): void
    {
        $draft = is_int($draft) ? ilForumPostDraft::newInstanceByDraftId($draft) : $draft;
        if (!$this->hasDraftAccess($draft)) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }
    }

    private function hasDraftAccess(int|ilForumPostDraft $draft): bool
    {
        $draft = is_int($draft) ? ilForumPostDraft::newInstanceByDraftId($draft) : $draft;
        return (
            $draft->getDraftId() &&
            !$this->user->isAnonymous() &&
            $this->access->checkAccess('add_reply', '', $this->object->getRefId()) &&
            $this->user->getId() === $draft->getPostAuthorId()
        );
    }

    private function doHistoryCheck(int $draftId): void
    {
        if (!ilForumPostDraft::isAutoSavePostDraftAllowed()) {
            return;
        }

        $this->checkDraftAccess($draftId);

        iljQueryUtil::initjQuery();
        $draftsFromHistory = ilForumDraftsHistory::getInstancesByDraftId($draftId);
        if ($draftsFromHistory !== []) {
            $form_tpl = new ilTemplate('tpl.restore_thread_draft.html', true, true, 'components/ILIAS/Forum');

            $first_open = null;
            foreach ($draftsFromHistory as $history_instance) {
                $accordion = new ilAccordionGUI();
                $accordion->setId('acc_' . $history_instance->getHistoryId());
                $accordion->setBehaviour(ilAccordionGUI::ALL_CLOSED);
                if ($first_open === null) {
                    $first_open = $history_instance->getHistoryId();
                    $accordion->setBehaviour(ilAccordionGUI::FIRST_OPEN);
                }

                $form_tpl->setCurrentBlock('list_item');
                $message = ilRTE::_replaceMediaObjectImageSrc($history_instance->getPostMessage(), 1);

                $history_date = ilDatePresentation::formatDate(new ilDateTime(
                    $history_instance->getDraftDate(),
                    IL_CAL_DATETIME
                ));
                $this->ctrl->setParameter($this, 'history_id', $history_instance->getHistoryId());
                $header = $history_date . ': ' . $history_instance->getPostSubject();

                $accordion_tpl = new ilTemplate(
                    'tpl.restore_thread_draft_accordion_content.html',
                    true,
                    true,
                    'components/ILIAS/Forum'
                );
                $accordion_tpl->setVariable('MESSAGE', $message);
                $accordion_tpl->setVariable(
                    'BUTTON',
                    $this->uiRenderer->render(
                        $this->uiFactory->button()->standard(
                            $this->lng->txt('restore'),
                            $this->ctrl->getLinkTarget($this, 'restoreFromHistory')
                        )
                    )
                );
                $accordion->addItem($header, $accordion_tpl->get());

                $form_tpl->setVariable('ACC_AUTO_SAVE', $accordion->getHTML());
                $form_tpl->parseCurrentBlock();
            }

            $form_tpl->setVariable('RESTORE_DATA_EXISTS', 'found_threat_history_to_restore');

            $modal = $this->ui_factory->modal()->interruptive(
                $this->lng->txt('restore_draft_from_autosave'),
                $form_tpl->get(),
                '#'
            )->withAdditionalOnLoadCode(function (string $id): string {
                return "document.getElementById('$id').dataset.modalId = 'frm_autosave_restore';";
            });

            $this->modal_history = $this->ui_renderer->render($modal);
        } else {
            ilForumPostDraft::createDraftBackup($draftId);
        }
    }

    private function renderPostingForm(ilTemplate $tpl, ilForum $frm, ilForumPost $node, string $action): void
    {
        $draft_id = $this->retrieveDraftId();
        if ($action === 'showedit' && (
            (!$this->is_moderator && !$node->isOwner($this->user->getId())) ||
                $this->user->isAnonymous() ||
                $node->isCensored()
        )) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        } elseif ($action === 'showreply' && !$this->access->checkAccess('add_reply', '', $this->object->getRefId())) {
            $this->error->raiseError($this->lng->txt('permission_denied'), $this->error->MESSAGE);
        }

        $tpl->setVariable('REPLY_ANKER', 'reply_' . $this->objCurrentPost->getId());
        $oEditReplyForm = $this->getReplyEditForm();
        $subject = '';
        if ($action !== 'editdraft') {
            switch ($this->objProperties->getSubjectSetting()) {
                case 'add_re_to_subject':
                    $subject = $this->getModifiedReOnSubject();
                    break;

                case 'preset_subject':
                    $subject = $this->objCurrentPost->getSubject();
                    break;

                case 'empty_subject':
                    $subject = '';
                    break;
            }
        }

        switch ($action) {
            case 'showreply':
                if ($this->ctrl->getCmd() === 'savePost' || $this->ctrl->getCmd() === 'saveAsDraft') {
                    $oEditReplyForm->setValuesByPost();
                } elseif ($this->ctrl->getCmd() === 'quotePost') {
                    $authorinfo = new ilForumAuthorInformation(
                        $node->getPosAuthorId(),
                        $node->getDisplayUserId(),
                        (string) $node->getUserAlias(),
                        (string) $node->getImportName()
                    );

                    $oEditReplyForm->setValuesByPost();
                    $oEditReplyForm->getItemByPostVar('message')->setValue(
                        ilRTE::_replaceMediaObjectImageSrc(
                            $frm->prepareText(
                                $node->getMessage(),
                                1,
                                $authorinfo->getAuthorName()
                            ) . "\n" . $oEditReplyForm->getInput('message'),
                            1
                        )
                    );
                } else {
                    $oEditReplyForm->setValuesByArray([
                        'draft_id' => $draft_id,
                        'alias' => '',
                        'subject' => $subject,
                        'message' => '',
                        'notify' => 0,
                        'userfile' => '',
                        'del_file' => []
                    ]);
                }
                break;

            case 'showedit':
                if ($this->ctrl->getCmd() === 'savePost') {
                    $oEditReplyForm->setValuesByPost();
                } else {
                    $oEditReplyForm->setValuesByArray([
                        'alias' => '',
                        'subject' => $this->objCurrentPost->getSubject(),
                        'message' => ilRTE::_replaceMediaObjectImageSrc($frm->prepareText(
                            $this->objCurrentPost->getMessage(),
                            2
                        ), 1),
                        'notify' => $this->objCurrentPost->isNotificationEnabled() && !$this->user->isAnonymous(),
                        'userfile' => '',
                        'del_file' => [],
                        'draft_id' => $draft_id
                    ]);
                }
                break;

            case 'editdraft':
                if (in_array($this->ctrl->getCmd(), ['saveDraft', 'updateDraft', 'publishDraft'])) {
                    $oEditReplyForm->setValuesByPost();
                } elseif ($draft_id > 0) {
                    $draftObject = new ilForumPostDraft(
                        $this->user->getId(),
                        $this->objCurrentPost->getId(),
                        $draft_id
                    );
                    $oEditReplyForm->setValuesByArray([
                        'alias' => $draftObject->getPostUserAlias(),
                        'subject' => $draftObject->getPostSubject(),
                        'message' => ilRTE::_replaceMediaObjectImageSrc($frm->prepareText(
                            $draftObject->getPostMessage(),
                            2
                        ), 1),
                        'notify' => $draftObject->isNotificationEnabled() && !$this->user->isAnonymous(),
                        'userfile' => '',
                        'del_file' => [],
                        'draft_id' => $draft_id
                    ]);
                }
                break;
        }

        $this->ctrl->setParameter($this, 'pos_pk', $this->objCurrentPost->getId());
        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentPost->getThreadId());
        $this->ctrl->setParameter($this, 'page', (int) ($this->httpRequest->getQueryParams()['page'] ?? 0));
        $this->ctrl->setParameter(
            $this,
            'orderby',
            $this->getOrderByParam()
        );
        $this->ctrl->setParameter(
            $this,
            'action',
            ilUtil::stripSlashes($this->requestAction)
        );
        if ($action !== 'editdraft') {
            $tpl->setVariable('FORM', $oEditReplyForm->getHTML());
        }
        $this->ctrl->clearParameters($this);
    }

    private function getResetLimitedViewInfo(): string
    {
        $this->ctrl->setParameter($this, 'thr_pk', $this->objCurrentTopic->getId());

        $buttons = [
            $this->uiFactory->button()->standard(
                $this->lng->txt('reset_limited_view_button'),
                $this->ctrl->getLinkTarget($this, 'resetLimitedView')
            )
        ];

        return $this->uiRenderer->render(
            $this->uiFactory
                ->messageBox()
                ->info($this->lng->txt('reset_limited_view_info'))
                ->withButtons($buttons)
        );
    }

    private function getOrderByParam(): string
    {
        return ilUtil::stripSlashes($this->http->wrapper()->query()->retrieve(
            'orderby',
            $this->refinery->byTrying([
                $this->refinery->kindlyTo()->string(),
                $this->refinery->always('')
            ])
        ));
    }
}
