<?php

/**
 * This file is part of ILIAS, a powerful learning management system
 * published by ILIAS open source e-Learning e.V.
 *
 * ILIAS is licensed with the GPL-3.0,
 * see https://www.gnu.org/licenses/gpl-3.0.en.html
 * You should have received a copy of said license along with the
 * source code, too.
 *
 * If this is not the case or you just want to try ILIAS, you'll find
 * us at:
 * https://www.ilias.de
 * https://github.com/ILIAS-eLearning
 *
 *********************************************************************/

declare(strict_types=1);

use ILIAS\ResourceStorage\Identification\ResourceIdentification;
use ILIAS\ResourceStorage\Collection\ResourceCollection;

use ILIAS\ResourceStorage\Revision\Revision;

use ILIAS\UI\Component\Input\Container\Form\Standard;

use ILIAS\ResourceStorage\Flavour\Definition\PagesToExtract;

/**
 * @author Fabian Schmid <fabian@sr.solutions>
 */
class ilFileHandlingDemoGUI implements ilCtrlBaseClassInterface
{
    private const CMD_INDEX = 'index';
    private const CMD_ADD = 'add';
    private const CMD_CREATE = 'create';
    private const CMD_DOWNLOAD = 'download';
    private const CMD_SHOW_ZIP = 'showZip';
    private const CMD_ADD_TO_COLLECTION = 'addToCollection';
    private const CMD_DOWNLOAD_COLLECTION = 'downloadCollection';
    private const CMD_DELETE = 'delete';
    private const CMD_SHOW = 'show';
    private const CMD_SHOW_PDF = 'showPDF';

    private const F_FILE = 'file';
    private const P_RID = 'rid';
    private const DEMO_COLLECTION_ID = 'demo_collection_id';

    private ilCtrlInterface $ctrl;
    private \ILIAS\HTTP\Services $http;
    private \ILIAS\FileUpload\FileUpload $upload;
    private \ILIAS\Filesystem\Filesystem $filesystem;
    private \ILIAS\ResourceStorage\Services $irss;
    private \ILIAS\Filesystem\Util\Archive\Archives $archives;
    private \ILIAS\Refinery\Factory $refinery;

    public function __construct()
    {
        global $DIC;
        // General Dependencies
        $this->ctrl = $DIC->ctrl();
        $this->http = $DIC->http();
        $this->refinery = $DIC->refinery();
        $this->ui = $DIC->ui();
        // ilSession::set(self::DEMO_COLLECTION_ID, null);

        // File-Handling Dependencies
        $this->upload = $DIC->upload();
        $this->filesystem = $DIC->filesystem()->storage();
        $this->irss = $DIC->resourceStorage();
        $this->archives = $DIC->archives();
    }

    public function executeCommand(): void
    {
        $this->ui->mainTemplate()->loadStandardTemplate();
        $this->ui->mainTemplate()->setTitle(
            'File Handling Demo'
        ); // Something like $this->lng->txt('file_handling_demo') should be used here, but for the Demo we use strings directly

        switch ($this->ctrl->getNextClass()) {
            case strtolower(ilFileHandlingDemoUploadHandlerGUI::class):
                $upload_handler = new ilFileHandlingDemoUploadHandlerGUI();
                $this->ctrl->forwardCommand($upload_handler);
                break;
            default:
                $this->performCommand();
                break;
        }
        $this->ui->mainTemplate()->printToStdout();
    }

    protected function performCommand(): void
    {
        switch ($this->ctrl->getCmd(self::CMD_ADD)) {
            case self::CMD_DOWNLOAD:
                $this->download();
                break;
            case self::CMD_SHOW:
                $this->show();
                break;
            case self::CMD_INDEX:
                $this->index();
                break;
            case self::CMD_ADD:
                $this->add();
                break;
            case self::CMD_DELETE:
                $this->delete();
                break;
            case self::CMD_CREATE:
                $this->create();
                break;
            case self::CMD_ADD_TO_COLLECTION:
                $this->addToCollection();
                break;
            case self::CMD_SHOW_ZIP:
                $this->showZip();
                break;
            case self::CMD_SHOW_PDF:
                $this->showPDF();
                break;
        }
    }

    protected function buildForm(): Standard
    {
        // Create Upload-Form
        $upload_handler = new ilFileHandlingDemoUploadHandlerGUI();
        return $this->ui->factory()->input()->container()->form()->standard(
            $this->ctrl->getLinkTarget($this, self::CMD_CREATE),
            [
                self::F_FILE => $this->ui->factory()->input()->field()->file(
                    $upload_handler,
                    'File',
                    'Upload a File'
                )
            ]
        );
    }

    private function add(): void
    {
        $form = $this->buildForm();
        $this->ui->mainTemplate()->setContent(
            $this->ui->renderer()->render($form)
        );
    }

    private function create(): void
    {
        $form = $this->buildForm();
        $form = $form->withRequest($this->http->request());
        $data = $form->getData()[self::F_FILE] ?? [];

        // At this place we want to store the resourse identification string (RID) in the database and use it later to retrieve the file.
        // Since this is just a demo, we just print the RIS to the screen and make a download button which has the RID as e parameter.

        $resource_id_string = $data[0] ?? '';
        $this->ctrl->setParameter($this, self::P_RID, $resource_id_string);

        $download_button = $this->ui->factory()->button()->primary(
            'Show File',
            $this->ctrl->getLinkTarget($this, self::CMD_SHOW)
        );

        $this->ui->mainTemplate()->setContent(
            '<pre>' . print_r($data, true) . '</pre>'
            . $this->ui->renderer()->render($download_button)
        );
    }

    private function show(): void
    {
        // Retrieve the RID from the URL
        $rid = $this->ridFromRequest();

        $some_infos_of_the_resource_to_show = $this->ridToArray($rid);

        // Buttons
        $this->ctrl->saveParameter($this, self::P_RID);

        $buttons = [];

        $buttons['add_to_collection'] = $this->ui->factory()->button()->primary(
            'Add File to Collection',
            $this->ctrl->getLinkTarget($this, self::CMD_ADD_TO_COLLECTION)
        );

        $buttons['delete'] = $this->ui->factory()->button()->standard(
            'Delete File',
            $this->ctrl->getLinkTarget($this, self::CMD_DELETE)
        );

        $buttons['download'] = $this->ui->factory()->button()->standard(
            'Download File',
            $this->ctrl->getLinkTarget($this, self::CMD_DOWNLOAD)
        );

        // If file is a ZIP, we can show the content of the ZIP with the following button
        $current_revision = $this->ridToCurrentRevision($rid);
        if ($current_revision->getInformation()->getMimeType() === 'application/zip'
            || $current_revision->getInformation()->getMimeType() === 'application/x-zip-compressed'
        ) {
            $buttons['show_zip'] = $this->ui->factory()->button()->standard(
                'Show ZIP Content',
                $this->ctrl->getLinkTarget($this, self::CMD_SHOW_ZIP)
            );
        }

        // if the file is a PDF, we can show it with the following button
        if ($current_revision->getInformation()->getMimeType() === 'application/pdf') {
            $buttons['show_pdf'] = $this->ui->factory()->button()->standard(
                'Show PDF',
                $this->ctrl->getLinkTarget($this, self::CMD_SHOW_PDF)
            );
        }

        $this->ui->mainTemplate()->setContent(
            '<pre>' . print_r($some_infos_of_the_resource_to_show, true) . '</pre>'
            . $this->ui->renderer()->render($buttons)
        );
    }

    private function addToCollection(): void
    {
        $rid = $this->ridFromRequest();

        $collection = $this->getCollection();
        $collection->add($rid);
        $return = $this->irss->collection()->store($collection);

        $this->ui->mainTemplate()->setOnScreenMessage('success', 'File added to Collection');

        $show = $this->ui->factory()->button()->standard(
            'Show Collection',
            $this->ctrl->getLinkTarget($this, self::CMD_INDEX)
        );

        $this->ui->mainTemplate()->setContent(
            $this->ui->renderer()->render([$show])
        );
    }

    private function download(): void
    {
        $rid = $this->ridFromRequest();

        $this->irss->consume()->download($rid)->run();
    }

    private function delete(): void
    {
        $rid = $this->ridFromRequest();

        // Delete the Resource
        $this->irss->manage()->remove($rid, new ilFileHandlingDemoStakeholder());
        $this->ui->mainTemplate()->setOnScreenMessage('success', 'File deleted', true);
        $this->ctrl->redirect($this, self::CMD_ADD);
    }

    private function index(): void
    {
        $collection = $this->getCollection();

        $collection_info = array_map(function (ResourceIdentification $rid): array {
            return $this->ridToArray($rid);
        }, $collection->getResourceIdentifications());

        $add = $this->ui->factory()->button()->primary(
            'Add New File',
            $this->ctrl->getLinkTarget($this, self::CMD_ADD)
        );

        $download_button = $this->ui->factory()->button()->standard(
            'Download Collection',
            $this->ctrl->getLinkTarget($this, self::CMD_DOWNLOAD_COLLECTION)
        );

        $this->ui->mainTemplate()->setContent(
            '<pre>' . print_r($collection_info, true) . '</pre>'
            . $this->ui->renderer()->render([$add, $download_button])
        );
    }

    private function showPDF(): void
    {
        $rid = $this->ridFromRequest();

        $flavour = $this->irss->flavours()->get(
            $rid,
            new PagesToExtract(false, 400, 10, false)
        );

        $images = array_map(function (string $url): string {
            return '<img src="' . $url . '" />';
        }, $this->irss->consume()->flavourUrls($flavour)->getURLsAsArray());

        $this->ctrl->saveParameter($this, self::P_RID);
        $back = $this->ui->factory()->button()->standard(
            'Back',
            $this->ctrl->getLinkTarget($this, self::CMD_SHOW)
        );

        $this->ui->mainTemplate()->setContent(
            $this->ui->renderer()->render($back) . '<br>'
            . implode(' ', $images)
        );
    }

    private function showZip(): void
    {
        $rid = $this->ridFromRequest();
        $stream = $this->irss->consume()->stream($rid)->getStream();
        $unzip = $this->archives->unzip($stream);
        $content = iterator_to_array($unzip->getPaths());

        $this->ctrl->saveParameter($this, self::P_RID);
        $back = $this->ui->factory()->button()->standard(
            'Back',
            $this->ctrl->getLinkTarget($this, self::CMD_SHOW)
        );

        $this->ui->mainTemplate()->setContent(
            '<pre>' . print_r($content, true) . '</pre>'
            . $this->ui->renderer()->render($back)
        );
    }

    private function downloadCollection(): void
    {
        $collection = $this->getCollection();
        $this->irss->consume()->downloadCollection($collection)->run();
    }

    private function ridFromRequest(): ResourceIdentification
    {
        // Retrieve the RID from the URL
        $rid_string = $this->http->wrapper()->query()->has(self::P_RID)
            ? $this->http->wrapper()->query()->retrieve(self::P_RID, $this->refinery->to()->string())
            : throw new \InvalidArgumentException('No RID given');

        // Retrieve the Resource from the Resource Storage Service
        $rid = $this->irss->manage()->find($rid_string) ?? throw new \InvalidArgumentException(
            'No Resource found for RID: ' . $rid_string
        );
        return $rid;
    }

    protected function ridToArray(ResourceIdentification $rid): array
    {
        $current_revision = $this->ridToCurrentRevision($rid);

        $some_infos_of_the_resource_to_show = [
            'RID' => $current_revision->getIdentification()->serialize(),
            'Revision Title' => $current_revision->getTitle(),
            'Filename' => $current_revision->getInformation()->getTitle(),
            'Mime-Type' => $current_revision->getInformation()->getMimeType(),
            'Size' => $current_revision->getInformation()->getSize(),
            'Timestamp' => $current_revision->getInformation()->getCreationDate()->format('Y-m-d H:i:s')
        ];
        return $some_infos_of_the_resource_to_show;
    }

    protected function getCollection(): ResourceCollection
    {
        // In the real world, you would store the collection in the database and retrieve it from there. We use the session here for simplicity.
        $collection_id_string = ilSession::get(self::DEMO_COLLECTION_ID);
        if ($collection_id_string === null || !is_string($collection_id_string)) {
            $new_collection_id = $this->irss->collection()->id();
            $collection = $this->irss->collection()->get($new_collection_id);
            $this->irss->collection()->store($collection);
            $collection_id_string = $new_collection_id->serialize();
            ilSession::set(self::DEMO_COLLECTION_ID, $collection_id_string);
        }

        $identification = $this->irss->collection()->id($collection_id_string);

        $resource_collection = $this->irss->collection()->get(
            $identification
        );
        return $resource_collection;
    }

    protected function ridToCurrentRevision(ResourceIdentification $rid): Revision
    {
        $resource = $this->irss->manage()->getResource($rid);
        return $resource->getCurrentRevision();
    }
}
